/*
 *  fresh.c
 *
 *  Low-level support for freshness functionality.
 *
 *  (c) Copyright 2003-2004, Mark R. Shinwell.
 *  (c) Copyright 2005, INRIA [Francesco Zappa Nardelli]
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <caml/fail.h>
#include <caml/alloc.h>
#include <caml/memory.h>
#include <caml/custom.h>
#include <caml/mlvalues.h>
#include <caml/callback.h>

#include "hashtable_cwc22.h" 

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/***************************************************************************
 * Functions local to this module.
 **************************************************************************/

#define Is_name(name_tag, v) (Is_block(v) && Tag_val(v) == 0 &&\
                              Wosize_val(v) == 2 && Field(v,1) == name_tag)


static int atom_id = 0; /* FIXME needs to go to 64 bits */

static unsigned int value_hasher(void* v)
{
    /* Hash function for things of type "value". */

    return (unsigned int) v; 
}

static int value_equality(void* v1, void* v2)
{
    /* Hashtable equality function for things of type "value". */

    return v1 == v2;
}

/* Allocate a new atom. */
static value alloc_atom(value name_tag)
{
    CAMLparam1(name_tag);
    CAMLlocal1(atom);

    /* an atom is a block with tag name_tag, whose first field
     * is an "atom identifier".
     */
    /* an atom is a block with tag 0 and size 2, whose first field is a name_tag,
     * the second, is an "atom identifier".
     */

    atom = caml_alloc(2, 0);
    Store_field(atom, 0, Val_int(atom_id++));
    Store_field(atom, 1, name_tag);

    CAMLreturn(atom);
}

typedef value (*alloc_fn)(int size, int tag);

typedef enum {
    entry_ALLOCATE,
    entry_SHARED
} dynarray_entry_type;

typedef struct {
    dynarray_entry_type type;
    union {
        struct {
            alloc_fn fn;
            int size;
            int tag;
        } allocate;
        struct {
            int index;
        } shared;
    } data;
} dynarray_entry;

typedef struct {
    dynarray_entry* data;
    int allocated;
    int used;
} dynarray;

static dynarray dynarray_alloc(void)
{
    dynarray da;
    da.allocated = 16;
    da.used = 0;
    da.data = (dynarray_entry*) malloc(da.allocated * sizeof(dynarray_entry));
    return da;
}

static void dynarray_free(dynarray da)
{
    free((void*) da.data);
}

static void dynarray_check_size(dynarray* da)
{
    if (da->used == da->allocated) {
        da->allocated *= 2;
        da->data = (dynarray_entry*)
          realloc((void*) da->data, da->allocated * sizeof(dynarray_entry));
    }
}

static int dynarray_add_allocation(dynarray* da,
                                   alloc_fn fn, int size, int tag)
{
    dynarray_check_size(da);
    
    da->data[da->used].type = entry_ALLOCATE;
    da->data[da->used].data.allocate.fn = fn;
    da->data[da->used].data.allocate.size = size;
    da->data[da->used].data.allocate.tag = tag;
    da->used++;

    return da->used - 1;
}

static void dynarray_add_shared(dynarray* da, int index)
{
    assert(index < da->used);
    
    dynarray_check_size(da);

    da->data[da->used].type = entry_SHARED;
    da->data[da->used].data.shared.index = index;
    da->used++;
}

static value dynarray_to_blocks(dynarray* da)
{
    CAMLparam0();
    CAMLlocal1(v);

    v = Val_unit;

    if (da->used > 0) {
        int i;
        v = caml_alloc_tuple(da->used);
        for (i = 0; i < da->used; i++) {
            if (da->data[i].type == entry_ALLOCATE) {
                Store_field(v, i,
                            (*(da->data[i].data.allocate.fn))
                            (da->data[i].data.allocate.size,
                             da->data[i].data.allocate.tag));
            }
            else {
                assert(i > 0);
                assert(da->data[i].data.shared.index < i);

                Store_field(v, i, Field(v, da->data[i].data.shared.index));
            }
        }
    }

    CAMLreturn(v);
}

static int dynarray_is_block_shared(dynarray* da, int index)
{
    /* Returns non-zero if the block at "index" is shared. */

    assert(da);
    assert(index >= 0 && index < da->used);

    return (da->data[index].type == entry_SHARED);
}

static value alloc_fn_gen(int size, int tag)
{
    return caml_alloc(size, tag);
}

static void deep_copy_collect_sizes(value name_tag, dynarray* da, 
                                    struct hashtable* ht, value v)
{
  /* No allocation on the ML heap may happen within this function, or
   * "ht" will potentially be invalidated.
   */
  
  /* The hashtable maps from values which we have seen and will need to
   * be copied, to integers specifying the *1-based* index in the array
   * "da" corresponding to that value.  We have to use a 1-based index
   * to distinguish the "not found" return value from hashtable_search.
   */
  if (Is_block(v) && (Tag_val(v) < No_scan_tag &&
                      Tag_val(v) != Object_tag &&
                      !Is_name(name_tag, v))) {
    void* data = hashtable_search(ht, (void*) v);

    if (data) {
      /* seen this value already: mark it as a shared block in
       * the array */
      dynarray_add_shared(da, ((int) data) - 1);
      return;
    }
    else {
      int x;
      
      hashtable_insert(ht, (void*) v,
                       (void*) (1 + dynarray_add_allocation(da, alloc_fn_gen, Wosize_val(v), Tag_val(v))));
      for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
           x < Wosize_val(v);
           x++) {
        deep_copy_collect_sizes(name_tag, da, ht, Field(v, x));
      }
    }
  }
}

static value deep_copy(value name_tag, value blocks, int* next_block,
                       dynarray* da, value v,
                       value atoms_1, value atoms_2)
{
  CAMLparam5(name_tag, blocks, v, atoms_1, atoms_2);
  CAMLlocal1(ret);
  
  assert(Is_block(atoms_1) && Tag_val(atoms_1) == 0);
  assert(Is_block(atoms_2) && Tag_val(atoms_2) == 0);
  assert(Wosize_val(atoms_1) == Wosize_val(atoms_2));
  
  ret = v;
  
  if (Is_block(v) && (Tag_val(v) < No_scan_tag &&
                      Tag_val(v) != Object_tag &&
                      !Is_name(name_tag, v))) {
    CAMLlocal1(data);
    
    assert(Is_block(blocks));
    assert(Wosize_val(blocks) == da->used);
    assert(blocks != Val_unit);
    assert(*next_block < da->used);
    
    ret = Field(blocks, *next_block);
    *next_block = *next_block + 1;
    
    if (!dynarray_is_block_shared(da, *next_block - 1)) {
      int x;
      
      /* copy code pointer for closures */
      if (Tag_val(v) == Closure_tag) {
        Store_field(ret, 0, Field(v, 0));
      }
      
      /* copy the rest */
      for (x = (Tag_val(v) == Closure_tag ? 1 : 0);
           x < Wosize_val(v);
           x++) {
        Store_field(ret, x,
                    deep_copy(name_tag, blocks, next_block, da, 
                              Field(v, x), atoms_1, atoms_2));
      }
    }
  }
  else if (Is_name(name_tag, v)) {
    /* Apply any swap which might apply to this atom.
     * We're actually always going to be doing fresh renaming,
     * so there will be at most one swap which affects any particular atom.
     */
    int x;
    int done = 0;
    for (x = 0; !done && x < Wosize_val(atoms_1); x++) {
      if (Field(v, 0) == Field(Field(atoms_1, x), 0)) {
        ret = Field(atoms_2, x);
        done = 1;
      }
    }
    for (x = 0; !done && x < Wosize_val(atoms_1); x++) {
      if (Field(v, 0) == Field(Field(atoms_2, x), 0)) {
        ret = Field(atoms_1, x);
        done = 1;
      }
    }
  }
  CAMLreturn(ret);
}

/* Given two equally-sized tuples of atoms, swap the Nth element
 * of the first tuple and the Nth element of the second tuple
 * throughout the value v, copying the value along the way.
 * At most one swap must apply to any particular atom in v.
 */
static value swap_atoms(value name_tag, value atoms_1, value atoms_2, value v)
{
  CAMLparam4(name_tag, atoms_1, atoms_2, v);
  CAMLlocal2(ret, blocks);
  dynarray da;
  struct hashtable* ht;
  int next_block = 0;
  
  /* This needs to be done in multiple stages to avoid the necessity
   * of having hashtables keyed on "value"s which are stable under GC.
   */
  
  /* Stage 1: collect sizes of blocks needed for a copy of "v"
   * (no ML allocation is allowed in this stage as we require a hashtable
   *  to detect cyclic structures)
   */
  da = dynarray_alloc();
  ht = create_hashtable(64, 0.75, value_hasher, value_equality);
  deep_copy_collect_sizes(name_tag, &da, ht, v);
  hashtable_destroy(ht, 0);
  
  /* Stage 2: allocate blocks */
  blocks = dynarray_to_blocks(&da);
  
  /* Stage 3: perform the copy */
  ret = deep_copy(name_tag, blocks, &next_block, &da, v, atoms_1, atoms_2);
  dynarray_free(da);
  
  CAMLreturn(ret);
}

/***************************************************************************
 * Exported primitives.
 **************************************************************************/

/* Allocate a new atom. */
CAMLprim value fresh_new_atom(value name_tag)
{
    CAMLparam1(name_tag);
    CAMLreturn(alloc_atom(name_tag));
}

/* Swap two atoms throughout a value. */
CAMLprim value fresh_swap_atoms(value name_tag, value atom1, value atom2, value v)
{
    CAMLparam4(name_tag,atom1, atom2, v);
    CAMLlocal2(blk1, blk2);
    CAMLlocal1(ret);

    blk1 = caml_alloc_tuple(1);
    blk2 = caml_alloc_tuple(1);
    Store_field(blk1, 0, atom1);
    Store_field(blk2, 0, atom2);
    
    ret = swap_atoms(name_tag, blk1, blk2, v);

    CAMLreturn(ret);
}

/* Swap multiple atoms pairwise throughout a value. */
CAMLprim value fresh_swap_multiple_atoms(value name_tag, value atoms1, value atoms2, value v)
{
    int i;
    int total;
    CAMLparam4(name_tag, atoms1, atoms2, v);
    CAMLlocal4(blk1, blk2, atoms1_copy, atoms2_copy);
    CAMLlocal1(ret);

    if (!((Is_block(atoms1) && (Is_block(atoms2))))) {
      raise_constant(*caml_named_value("arity_mismatch")); }

    total = 0;
    atoms1_copy = atoms1;
    atoms2_copy = atoms2;

    while (Is_block(atoms1)) {
        assert(Is_block(atoms2));

        total++;

        atoms1 = Field(atoms1, 1);
        atoms2 = Field(atoms2, 1);
    }

    assert(total > 0);

    atoms1 = atoms1_copy;
    atoms2 = atoms2_copy;

    blk1 = caml_alloc_tuple(total);
    blk2 = caml_alloc_tuple(total);
    for (i = 0; i < total; i++) {
        assert(Is_block(atoms1));
        assert(Is_block(atoms2));

        Store_field(blk1, i, Field(atoms1, 0));
        Store_field(blk2, i, Field(atoms2, 0));
        atoms1 = Field(atoms1, 1);
        atoms2 = Field(atoms2, 1);
    }
    
    ret = swap_atoms(name_tag, blk1, blk2, v);

    CAMLreturn(ret);
}
