(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

(* timeout *)

(* exception Timeout *)

(* let sigalrm_handler = Sys.Signal_handle (fun _ -> raise Timeout)  *)

(* let timeout f time = *)
(*   let old_behavior = Sys.signal Sys.sigalrm sigalrm_handler in *)
(*   let reset_sigalrm () = Sys.set_signal Sys.sigalrm old_behavior  *)
(*   in ignore (Unix.alarm time) ; *)
(*   try  let res = f () in reset_sigalrm () ; res   *)
(*   with exc -> reset_sigalrm (); raise exc *)


exception Timeout

let notimer ={ Unix.it_interval=0.0 ; Unix.it_value = 0.0 }

let handler = Sys.Signal_handle
  (fun _ ->
    Sys.set_signal Sys.sigalrm Sys.Signal_default ;
    ignore (Unix.setitimer Unix.ITIMER_REAL notimer);
    raise Timeout)

let timeout f time =
  Sys.set_signal Sys.sigalrm handler;
  let timer = { Unix.it_interval= 0. ; Unix.it_value = time } in
  ignore (Unix.setitimer Unix.ITIMER_REAL timer);
  let res =
    try
      f ()
    with
      | e ->
        ignore (Unix.setitimer Unix.ITIMER_REAL notimer);
        raise e
  in
  ignore (Unix.setitimer Unix.ITIMER_REAL notimer);
  res


(* i/o *)

let error s =
  print_endline ("error: "^s);
  exit 1

let warning s =
  print_endline ("warning: "^s)

let debug s =
  if false
  then print_endline s

let strip_zeros s =
  let rec count i = match s.[i] with
    | '0' -> count (i+1)
    | _   -> i
  in
  let num = count 0 in
  String.sub s num ((String.length s)-num)

(* types *)

let the s = match s with | Some s -> s | None -> error "internal: the"

(* lists *)

let rec last l =
  match l with
    | [] -> error "internal: last"
    | h::[] -> h
    | h::t -> last t

let take_n_elem n l =
  let rec aux n l b =
    match n,l with
      | 0,t -> (List.rev b,t)
      | _,[] -> error "internal: take_n_elem"
      | n,h::t -> aux (n-1) t (h::b)
  in aux n l []

(* exec *)

let opt_debug_exec = ref false

let exec s =
  if !opt_debug_exec then print_endline ("exec: "^s);
  Sys.command s

let exec_fail s =
  if exec s != 0
  then error ("command \""^s^"\" failed.")

let exec_timeout t s =
  match exec ("timeout "^(string_of_int t)^" "^s) with
    (* various error codes returned by the shell if timeout aborts *)
    | 143 | 255 | 124 | 137 -> None
    | r -> Some r

(* command line util *)

let rec combine l res =
  match l with
    | [] -> res
    | _ :: [] ->
      error "number of .log files must be even"
    | f1::f2::t ->
      combine t (((f1,f2),None)::res)
      (*We also need to pass on the type of file from where the log was generated*)
      (*which we don't know for the log files passed through command line*)

(* output formatting *)

let align s n =
  let rec add_n_space n s =
    if n = 0 then s
    else add_n_space (n-1) (s^" ") in
  if String.length s >= n then s else
  let i = n - (String.length s) in
  add_n_space i s

let strip ?(chars=" \t\r\n") s =
  let p = ref 0 in
  let l = String.length s in
  while !p < l && String.contains chars s.[!p] do
    incr p;
  done;
  let p = !p in
  let l = ref (l - 1) in
  while !l >= p && String.contains chars s.[!l] do
    decr l;
  done;
  String.sub s p (!l - p + 1)

(* padding *)

let pad n s =
  let m = n - String.length s in
   if m>=1 then s ^ String.make m ' ' else s

let pad2 n s s' =
  let m = n - String.length s in
   if m>=1 then s' ^ String.make m ' ' else s'

(* lists *)

let rec opt_map f l =
  match l with
    | None :: t -> opt_map f t
    | Some e :: t -> (f e) :: (opt_map f t)
    | [] -> []

let rec option_map f l =
  match l with
    | [] -> []
    | h :: t ->
      match f h with None -> option_map f t
	| Some e -> e :: (option_map f t)

let list_to_pair l =
  match l with
    | l1 :: l2 :: [] -> Some (l1, l2)
    | _ -> None

let remove_duplicates l =
  let sl = List.sort compare l in
  let rec aux l acc = match l with
    | [] -> List.rev acc
    | [x] -> List.rev (x::acc)
    | (x1::x2::xs) ->
      if x1 = x2
      then aux (x2::xs) acc
      else aux (x2::xs) (x1::acc)
  in aux sl []

(* split a csmith generated file *)

let split_main_old name =
  let fd_in = open_in name in
  let (fd_out_body,fd_out_main) =
    let n = Filename.chop_extension name in
    (open_out (n^"_body.c"), open_out (n^"_main.c")) in

  let is_func_decl line =
    try
      ignore (Str.search_forward (Str.regexp_string " func") line 0);
      if line.[(String.length line)-1] = ';' then Some true else Some false
    with Not_found -> None in

  let is_main line =
    try ignore (Str.search_forward (Str.regexp_string "main") line 0); true
    with Not_found -> false in
  let is_sink line =
    try ignore (Str.search_forward (Str.regexp_string "csmith_sink") line 0); true
    with Not_found -> false in

  let rec aux_body last_line fdcl rb =
    try
      let line = input_line fd_in in
      if is_main line
      then begin
	output_string fd_out_main "#include \"csmith.h\"\n";
	output_char fd_out_main '\n';
	List.iter (fun l -> output_string fd_out_main ("extern "^l^"\n")) fdcl;
	output_string fd_out_main last_line; output_char fd_out_main '\n';
	output_string fd_out_main line; output_char fd_out_main '\n';
	aux_main ()
      end else begin
	output_string fd_out_body last_line; output_char fd_out_body '\n';
	match is_func_decl line, rb with
	  | None, _ -> aux_body line fdcl rb
	  | Some true, false -> aux_body line (line::fdcl) rb
	  | Some true, true -> aux_body line fdcl rb
	  | Some false, _ -> aux_body line fdcl true
      end
    with End_of_file -> error "aux_body: main not found"
  and aux_main () =
    try
      let line = input_line fd_in in
      if is_sink line
      then aux_main ()
      else begin
	output_string fd_out_main line; output_char fd_out_main '\n';
	aux_main ()
      end
    with End_of_file -> close_in fd_in; close_out fd_out_body; close_out fd_out_main
  in
  aux_body "" [] false


let split_main_old_2 name =
  let fd_in = open_in name in
  let (fd_out_body,fd_out_main) =
    let n = Filename.chop_extension name in
    (open_out (n^"_body.c"), open_out (n^"_main.c")) in

  let is_start_forward line =
    try ignore (Str.search_forward (Str.regexp_string "FORWARD DECLARATIONS") line 0); true
    with Not_found -> false in
  let is_end_forward line =
    try ignore (Str.search_forward (Str.regexp_string "FUNCTIONS") line 0); true
    with Not_found -> false in
  let is_main line =
    try ignore (Str.search_forward (Str.regexp_string "main") line 0); true
    with Not_found -> false in
  let is_sink line =
    try ignore (Str.search_forward (Str.regexp_string "csmith_sink") line 0); true
    with Not_found -> false in


  let rec aux_body last_line fdcl =
    try
      let line = input_line fd_in in
      if is_main line
      then begin
	output_string fd_out_main "#include \"csmith.h\"\n";
	List.iter
	  (fun l ->
	    output_string fd_out_main (if l="" then "" else ("extern "^l));
	    output_char fd_out_main '\n')
	  (List.rev fdcl);
	output_char fd_out_main '\n';
	output_string fd_out_main last_line; output_char fd_out_main '\n';
	output_string fd_out_main line; output_char fd_out_main '\n';
	aux_main ()
      end else
	if is_start_forward line
	then begin
	  output_string fd_out_body last_line; output_char fd_out_body '\n';
	  output_string fd_out_body line; output_char fd_out_body '\n';
	  aux_forward [];
	end else begin
	  output_string fd_out_body last_line; output_char fd_out_body '\n';
	  aux_body line fdcl
	end
    with End_of_file -> error "aux_body: main not found"
  and aux_main () =
    try
      let line = input_line fd_in in
      if is_sink line
      then aux_main ()
      else begin
	output_string fd_out_main line; output_char fd_out_main '\n';
	aux_main ()
      end
    with End_of_file -> close_in fd_in; close_out fd_out_body; close_out fd_out_main
  and aux_forward fdcl =
    try
      let line = input_line fd_in in
      if is_end_forward line
      then aux_body line fdcl
      else begin
	output_string fd_out_body line; output_char fd_out_body '\n';
	aux_forward (line::fdcl)
      end
    with End_of_file -> error "aux_forward: forward end not found"
  in

  aux_body "" []

(* HERE *)

(* val search_regex : string -> string -> int -> string*int
   Return a tuple containing all the substrings matching the given
   regular expression and the index where the first string was matched
*)
let search_regex regex_string str finish =
  let regex = Str.regexp regex_string in
  let index = ref (finish-1) in
  let result = ref "" in
  let loop = ref true in
  while !loop = true && !index > 0 do
    try
      index := Str.search_backward regex str !index;
      result := Str.matched_string str ^ "\n" ^ !result;
      index := !index - 1;
    with Not_found ->
      loop := false;
  done;
  (!result, !index)


(* val split_main_aux : string -> string*int
   Returns a tuple containing the main funtion and the index where
   the main function starts
*)

let split_main_aux str =
  try
    let regex_main = Str.regexp ("\\(^int[ \n\t\r]+main[^)]+)\\)\\|"
				 ^ "\\(^void[ \n\r\t]+main[^)]+)\\)") in
    let index = Str.search_forward regex_main str 0 in
    let str_main = ref (Str.matched_string str) in
    let count = ref 0 in
    let i = ref (index + String.length !str_main) in
    while String.get str !i <> '}' || !count <> 1 do
      if String.get str !i = '{' then count := !count + 1;
      if String.get str !i = '}' then count := !count - 1;
      str_main := !str_main ^ String.make 1 (String.get str !i);
      i := !i + 1
    done;
    str_main := !str_main ^ String.make 1 (String.get str !i);
    (!str_main, index);
  with Not_found ->
    ("/*main not found*/", String.length str)

let split_main name =

  (* val load_file : string -> string
     Return the contents of the file in a string *)
  let load_file f =
    let ic = open_in f in
    let n = in_channel_length ic in
    let s = String.create n in
    really_input ic s 0 n;
    close_in ic;
    (s) in

  (* val write_file : string -> string -> unit
     Write a string to a file *)
  let write_file f str =
    let oc = open_out f in
    output_string oc str;
    close_out oc in

  let contents = load_file name in

  (* search main *)
  let (str_main, finish1) = split_main_aux contents in
  (* search for function headers *)
  let (str_func, finish2) = search_regex (
    "^[a-z]+[^ \n\r\t;]+[^;{}]+func_[0-9]+[^\n)]+)[ \n\r\t]*{") contents finish1 in
  (* separate the function headers from their definitions *)
  let str_func_header = Str.global_replace (Str.regexp "[ \n\t\r]*{") ";" str_func in
  (* search the initialised global variables *)
  let (str_init_var, finish3) = search_regex (
    "\\(^[a-z]+[^;{}]+g_[0-9]+[^=;]*[ \n\t\r]*=[^;]+;\\)\\|"
    ^ "\\(^struct[ \n\t\r]+[^{;]*{[^}]+}[ \n\t\r]+g_[0-9]+[^;=]*=[^;]+;\\)\\|"
    ^ "\\(^union[ \n\t\r]+[^{;]*{[^}]+}[ \n\t\r]+g_[0-9]+[^;=]*=[^;]+;\\)\\|"
    ^ "\\(^[a-fh-z0-9_][a-z0-9_ ]+=[^;]+;\\)") contents finish2 in
  (* separate the variable headers from their definitions *)
  let str_init_var_header = Str.global_replace (Str.regexp "=[^;]+[ \n\t\r]*;") ";" str_init_var in
  (* search the uninitialised global variables *)
  let (str_uninit_var, finish4) = search_regex (
    "\\(^[a-z]+[^;{}=]+g_[0-9]+[^;=]*[ \n\t\r]*;\\)\\|"
    ^ "\\(^struct[ \n\t\r]+[^{;]*{[^}]+}[ \n\t\r]+g_[0-9]+[^;=]*;\\)\\|"
    ^ "\\(^union[ \n\t\r]+[^{;]*{[^}]+}[ \n\t\r]g_[0-9]+[^;=]*;\\)\\|"
    ^ "\\(^[a-fh-z0-9_][a-z0-9_ ]+;\\)") contents finish2 in
  write_file ((Filename.chop_extension name)^"_main.c") ("#include \"csmith.h\"\n" ^ str_uninit_var ^
				 "\n" ^ str_init_var_header ^ "\n" ^ str_func_header^"\n" ^ str_main);
  write_file ((Filename.chop_extension name)^"_body.c") (String.sub contents 0 finish1)

(* TO HERE *)

let count_lines filename =
  let fd = open_in filename in
  let rec aux n =
    let l =
      try Some (String.length (input_line fd))
      with End_of_file -> close_in fd; None
    in
    match l with
      | None -> n
      | Some l -> aux (n+l+1)
  in aux 0
