(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

open Util

type atomic_attribute = 
  | Acquire
  | Release
  | Relaxed
  | Seq_cst

let is_seq_cst = 
  function Seq_cst -> true | _ -> false

(* raw_traces, as generated by pin *)

type size = int

type raw_loc = Int64.t
type raw_value = Z.t

type raw_event = 
  | RLoad of raw_loc * raw_value * size
  | RStore of raw_loc * raw_value * size 
  | RFlush
  | RLock of raw_loc
  | RUnlock of raw_loc

type raw_trace = raw_event list

let dump_raw_trace t =
  let string_of_raw_event e =
    match e with 
      | RLoad (rl,rv,s)  -> Printf.sprintf "Load  : %Lx : %s : %d" rl (Z.format "$X" rv) s
      | RStore (rl,rv,s) -> Printf.sprintf "Store : %Lx : %s : %d" rl (Z.format "$X" rv) s
      | RFlush           -> "Flush"
      | RLock rl         -> Printf.sprintf "Lock  : %Lx" rl
      | RUnlock rl       -> Printf.sprintf "Unlock: %Lx" rl in 
  List.iter 
    (fun e -> print_endline (string_of_raw_event e))
    t

(* traces, after editing *)

type loc_attr = 
  | Arr_idx of int
  | Str_idx of string
  | Pad

(* we identify a location by *)
(* - the base variable, eg. g_34 (int, as entry in the symbol table) *)
(* - the actual offset - used in the overlap test in the reorderable check *)
(* - the original access infos, e.g. [4].f2[5] - used when matching traces *)

type loc = int * int * loc_attr list

(* this ignores the attributes, and might not be invariant wrt the optimisation level *)
let compare_locs (b1,o1,a1) (b2,o2,a2) =
  if b1 = b2 
  then 
    if o1 = o2 
    then 
      compare a1 a2
    else 
      if (o2-o1)<0 
      then 1 
      else -1 
  else 
    if (b2-b1)<0 
    then 1 
    else -1

type loc_bo = int * int

let loc_bo_of_loc (b,o,_) = (b,o)

type value = NonPointer of Z.t | Pointer of loc

type event = 
  | Init   of loc * value * size
  | Load   of loc * value * size
  | Store  of loc * value * size 
  | VLoad  of loc * value * size
  | VStore of loc * value * size 
  | ALoad  of atomic_attribute * loc * value * size
  | AStore of atomic_attribute * loc * value * size
  | Flush
  | Lock   of loc
  | Unlock of loc

let is_release e = 
  match e with
    | Init _ | Load _ | Store _ | VLoad _ | VStore _ 
      | AStore (Relaxed,_,_,_) | ALoad _ | Flush | Lock _ -> false
    | AStore _ | Unlock _ -> true

let is_acquire e = 
  match e with
    | Init _ | Load _ | Store _ | VLoad _ | VStore _ 
      | AStore _ | ALoad (Relaxed,_,_,_) | Flush | Unlock _ -> false
    | ALoad _ | Lock _ -> true

type access = R | W | I

let is_init e = 
  match e with
    | Init _ -> true
    | _ -> false

let is_na_read e = 
  match e with
    | Load _ | VLoad _ -> true
    | _ -> false

let is_atomic e =
  match e with
  | Init _ | Load _ | Store _ | VLoad _ | VStore _ -> false
  | ALoad _ | AStore _ | Flush | Lock _ | Unlock _ -> true

let is_relaxed e =
  match e with
    | ALoad (Relaxed,_,_,_) | AStore (Relaxed,_,_,_) -> true
    | _ -> false

type variable = string

type symbol_table =
    (variable * int) list * int

let empty_symbol_table = ([],0)

let add_symbol_table st var : symbol_table * int = 
  try (st, List.assoc var (fst st))
  with Not_found -> 
    let st,sti = st in 
    ((var,sti)::st, sti+1), sti
  
type rr =
  | IRR of int list
  | RAW | RAR 
  | OW of int list | WAR | WAW of int list
  | RSF (* redundant store flush, for release and relaxed store *) 
  | UIL (* reads from the .text section, for ARM only *)
  | CE  (* compund elimination, for the per-byte merging algorithm *)
  | SZ  (* enables a relaxed matching for the size of the events *)
  | NotRedundant 

let is_redundant rr =
  match rr with
    | NotRedundant -> false
    | _ -> true

let is_redundant_if_deps rr =
  match rr with
    | OW (_::_) | WAW (_::_) | IRR (_::_) -> true
    | _ -> false

type annot_event = { 
  evt         : event; 
  redundant   : rr ref;
  split_index : int;
}

type trace = {
  source     : string;
  executable : string;
  events     : event list;
  ir_reads   : int list;
  load_deps  : (int * int list) list;
  comp_opts  : string;
}

type trace_stats = {
  no_aload  : int;
  no_astore : int;
  no_vload  : int;
  no_vstore : int;
  no_load   : int;
  no_store  : int;
  no_flush  : int;
  no_lock   : int;
  no_unlock : int
}

(* tests *)

let same_loc (lb1,_,la1) (lb2,_,la2) =
  lb1 = lb2 && la1 = la2  (* TODO: avoid comparing strings in la *)

let is_struct (_,_,la) =
  List.exists (fun x -> match x with Str_idx _ -> true | _ -> false) la

let is_store = function
  | Store _ | VStore _ | AStore _ -> true
  | _ -> false

let loc_of_ev = function
  | Store (l,_,_) | Load (l,_,_) | VStore (l,_,_) | VLoad (l,_,_) 
    | AStore (_,l,_,_) | ALoad (_,l,_,_) 
    | Lock l | Unlock l | Init (l,_,_) -> Some l
  | Flush -> None

(* pretty print *)

type dump_traces = All | NoInit | Quiet
    
let string_of_loc_attr las =
  String.concat "" (List.map (fun la ->
    match la with
      | Arr_idx i -> "["^(string_of_int i)^"]"
      | Str_idx f -> "."^f
      | Pad -> " PAD"
  ) las)

let string_of_loc_base st lb =
  let rec aux st = 
    match st with
      | [] -> error 
	("malformed symbol table: couldn't find index "^(string_of_int lb))
      | (v,i)::t -> if i = lb then v else aux t
  in aux (fst st)

let string_of_location st (l:loc) =
  let (lb,lo,la) = l in
  if lb < 0 then ""
  else 
    (pad 12 ((string_of_loc_base st lb)^(string_of_loc_attr la))) 
    ^ (pad 5 ("<"^(string_of_int lo)^">"))

let string_of_loc_bo st (l:loc_bo) =
  let (lb,lo) = l in
  if lb < 0 then ""
  else 
    (pad 12 (string_of_loc_base st lb)) ^ (pad 5 ("<"^(string_of_int lo)^">"))

let string_of_value v st = 
  match v with 
  | Pointer x -> "&" ^ (string_of_location st x)
  | NonPointer s -> Z.format "%X" s

(* Int64.to_string s *)

let string_of_rr = function
  | NotRedundant -> "     "
  | IRR [] ->"IR"
  | IRR deps -> "IR {" ^ (String.concat "," (List.map string_of_int deps))^"}"
  | RAW ->   "RaW"
  | RAR ->   "RaR"
  | OW [] -> "OW"
  | OW deps -> "OW {" ^ (String.concat "," (List.map string_of_int deps))^"}"
  | WAR ->  "WaR"
  | WAW [] ->  "WaW"
  | WAW deps ->  "WaW {"^ (String.concat "," (List.map string_of_int deps))^"}"
  | RSF ->  "RsF" (*redundant store fence*)
  | UIL ->  "UIL"
  | CE ->  "CE"
  | SZ ->  "SZ"

let string_of_attribute = function
  | Acquire -> "acquire"
  | Release -> "release"
  | Relaxed -> "relaxed"
  | Seq_cst -> "seq_cst"

let string_of_lvs st l v s = 
  "   " 
  ^ Printf.sprintf "%s" (string_of_location st l) ^ " "
  ^ Printf.sprintf "%18s" (string_of_value v st) ^ " "
  ^ Printf.sprintf "%2s" (string_of_int s) 

let string_of_event st e = 
  let string_of_lvs = string_of_lvs st in
  match e with 
    | Init (l,v,s) -> 
        Printf.sprintf "%-10s %s" "Init" (string_of_lvs l v s)
    | Load (l,v,s) -> 
        Printf.sprintf "%-10s %s" "Load" (string_of_lvs l v s)
    | Store (l,v,s) -> 
        Printf.sprintf "%-10s %s" "Store" (string_of_lvs l v s)
    | VLoad (l,v,s) -> 
        Printf.sprintf "%-10s %s" "VLoad" (string_of_lvs l v s)
    | VStore (l,v,s) -> 
        Printf.sprintf "%-9s %s" "VStore" (string_of_lvs l v s)
    | ALoad (a,l,v,s) -> 
        Printf.sprintf "ALoad %-6s %s" (string_of_attribute a) (string_of_lvs l v s) 
    | AStore (a,l,v,s) -> 
        Printf.sprintf "AStore %-6s %s" (string_of_attribute a) (string_of_lvs l v s)  
    | Flush -> 
        Printf.sprintf "%s" "Flush"
    | Lock l -> 
        Printf.sprintf "%-10s    %-39s" "Lock" (string_of_location st l)
    | Unlock l -> 
        Printf.sprintf "%-10s    %-39s" "Unlock" (string_of_location st l)

let string_of_annot_event mode e st = 
  match e.evt,mode with
    | Init _, NoInit -> ""
    | Init (l,v,s), _ when (string_of_lvs st l v s).[3] = 'c' -> ""
    | _, _ -> 
      Printf.sprintf "%s %s %s %s"
        (if is_redundant !(e.redundant) then "*" else " ")
        (string_of_event st e.evt)
        (if e.split_index = 0 then "   " else Printf.sprintf "[%d]" e.split_index)
        (string_of_rr !(e.redundant))
 
let dump_symbol_table st =
  List.iter (fun (v,i) -> Printf.printf "%s -> %d\n" v i) (fst st)

let dump_trace mode t =
  let t,st = t in 
  List.iter 
    (fun e -> 
      match e, mode with
        | Init _, NoInit -> ()
        | Init (l,v,s), _ when (string_of_lvs st l v s).[3] = 'c' -> ()
        | _, _ ->
          print_endline (string_of_event st e))
    t

let dump_traces dump trace_unopt co_unopt trace_opt co_opt st = 
  if dump != Quiet then begin
    print_endline ("\n\n*** unoptimised trace: " ^ co_unopt ^"\n");
    dump_trace dump (trace_unopt,st);
    print_endline ("\n*** optimised trace: " ^ co_opt ^"\n");
    dump_trace dump (trace_opt,st);
    print_endline "\n***\n";
  end

let dump_annot_trace mode t =
  let t,st = t in 
  let rec aux t i =
    match t with
      | e::et ->
        let s = string_of_annot_event mode e st in
        if s = "" 
        then aux et i
        else 
          begin Printf.printf "%-70s   (%2d)\n" s i; 
            let i = match e.evt with | Init _ -> i | _ -> i+1 in
            aux et i
          end
      | [] -> ()
  in aux t 0

let dump_annot_traces dump trace_unopt co_unopt trace_opt co_opt st = 
  if dump != Quiet then begin
    print_endline ("\n\n*** unoptimised trace: " ^ co_unopt ^"\n");
    dump_annot_trace dump (trace_unopt,st);
    print_endline ("\n*** optimised trace: " ^ co_opt ^"\n");
    dump_annot_trace dump (trace_opt,st);
    print_endline "\n***\n";
  end;

(* mode *)

type itool = Valgrind | Pin

type compiler = Gcc | Clang | Icc

type filetype = C | Cpp

type arch = ARM | X86

(* *************************************** *)
(* global state for the stats in the paper *)

type stat_entry = {
  source_size : int;
  trace_size : int;
  no_eliminable : int;
  no_elim_details : (rr * int) list;
}

let no_tests = ref 200
let stats : stat_entry list ref = ref []

