(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

open Types
open Util

type delta = Delta | CReduce

(* csmith options *)
let opt_max_expr_complexity = ref 10
let opt_max_funcs = ref 10
let opt_atomics = ref false
let opt_locks = ref false
let opt_volatile = ref false
let opt_arrays = ref true
let opt_structs = ref true
let opt_pointers = ref true
let opt_csmith_seed = ref None
let opt_csmith_max_seed = ref None
let opt_csmith_swarm = ref false

(* cmmtest options *)
let opt_generate = ref None
let opt_baseline = ref false
let opt_trace_arch = ref None
let opt_dump_traces = ref Quiet
let opt_analyse = ref false
let opt_delta = ref None
let opt_asm = ref false
let opt_tool = ref Pin
let opt_compiler = ref Gcc
let opt_clean = ref false
let opt_quiet = ref false
let opt_real_quiet = ref false
let opt_timeout = ref (Some 30)
let opt_timeout_delta = ref false
let opt_repeat = ref false
let opt_repeat_forever = ref false
let opt_analyse_volatile_only = ref false
let opt_analyse_volatile_cached = ref false
let opt_analyse_count_only = ref false
let opt_analyse_stat = ref false
let opt_ir_unsound = ref true
let opt_ir_replay = ref false

(* compilation options *)
let opt_unopt = ref ""
let opt_opt = ref "-O2"

let c_files : string list ref = ref []
let log_files : ((string*string)*string option) list ref = ref []

let opts = Arg.align
  [("-generate",
    Arg.String
      (fun x -> match x with
      | "c" -> opt_generate := Some C
      | "cpp" -> opt_generate := Some Cpp
      | _ -> error "should be c or cpp only"),
    "<c|cpp> generate a c/cpp file via csmith");
   ("-expr_complexity",
    Arg.Int (fun i -> opt_max_expr_complexity := i),
    "<n> set Csmith expr_complexity to n");
   ("-timeout",
    Arg.Int (fun i -> if i != 0 then opt_timeout := Some i else ()),
    "<n> set timeout for analyse (in sec)");
   ("-max_funcs",
    Arg.Int (fun i -> opt_max_funcs := i),
    "<n> set Csmith max_funcs to n");
   ("-atomics",
    Arg.Unit (fun () -> opt_atomics := true),
    " generate code with low-level atomics");
   ("-locks",
    Arg.Unit (fun () -> opt_locks := true),
    " generate c code with locks");
   ("-volatile",
    Arg.Unit (fun () -> opt_volatile := true),
    " generate c code with volatiles");
   ("-trace", Arg.String (fun s ->
     match s with
       | "x86" -> opt_trace_arch := Some X86
       | "arm" -> opt_trace_arch := Some ARM
       | _ -> error "supported architectures are x86 and arm"),
    " trace the input files wrt the specified architecture");
   (* ("-tool", *)
   (*  Arg.String *)
   (*    (fun x -> match x with *)
   (*    | "valgrind" -> opt_tool := Valgrind *)
   (*    | "pin" -> opt_tool := Pin *)
   (*    | _ -> error "unknown tool"), *)
   (* "<pin|valgrind> specify how to instrument the binary"); *)
   ("-asm", Arg.Unit (fun () -> opt_asm := true),
    " generate the optimised and unoptimsed assembly");
   ("-dump_traces", Arg.Unit (fun () -> opt_dump_traces := NoInit),
    " dump traces");
   ("-dump_traces_init", Arg.Unit (fun () -> opt_dump_traces := All),
    " dump traces including init values");
   (* ("-baseline", Arg.Unit (fun () -> opt_baseline := true), *)
   (*  " compare the reference traces generated by gcc and llvm"); *)
   ("-analyse", Arg.Unit (fun () -> opt_analyse := true),
    " run the analyser on the generated file");
   ("-analyse_volatile_only", Arg.Unit (fun () -> opt_analyse_volatile_only := true),
    " analyse only volatile accesses");
   ("-analyse_volatile_cached", Arg.Unit (fun () -> opt_analyse_volatile_cached := true),
    " hunt for caching of volatile reads");
   (* ("-analyse_count_only", Arg.Unit (fun () -> opt_analyse_count_only := true), *)
   (*  " just compare the no. of atomic actions during analysis"); *)
   (* ("-statistics", Arg.Unit (fun () -> opt_analyse_stat := true), *)
   (*  " gather statistics about eliminable accesses"); *)
   ("-delta", Arg.String (fun s -> match s with
        | "delta" -> opt_delta := Some Delta
        | "c_reduce" -> opt_delta := Some CReduce
        | _ -> error "should be delta or c_reduce only"),
    " run delta if analyse fails");
   (* ("-timeout_delta", Arg.Unit (fun () -> opt_timeout_delta := true), *)
   (*  "consider a timeout as a failing analysis for the purpose of delta"); *)
   ("-clean", Arg.Unit (fun () -> opt_clean := true),
    " clean up if matching returns true");
   ("-quiet", Arg.Unit (fun () -> opt_quiet := true),
    " disable the gcc/llvm warnings and errors");
   ("-real_quiet", Arg.Unit (fun () -> opt_quiet := true; opt_real_quiet := true),
    " compact output");
   ("-repeat", Arg.Unit (fun () -> opt_repeat := true),
    " compatible only with -generate, repeats until the analyse
       returns false for the generated program");
   ("-repeat_forever", Arg.Unit (fun () -> opt_repeat := true; opt_repeat_forever := true),
    " compatible only with -generate, repeats forever");
   ("-no_unsound_ir", Arg.Unit (fun () -> opt_ir_unsound := false),
    " ignore loads used in jumps and do not mark as relevant");
   ("-ir_replay", Arg.Unit (fun () -> opt_ir_replay := true),
    " mark ir loads in the optimised trace by the replay mechanism");
   ("-no_arrays", Arg.Unit (fun () -> opt_arrays := false),
    " disable arrays when generating programs using csmith");
   ("-no_structs", Arg.Unit (fun () -> opt_structs := false),
    " disable structs when generating programs using csmith");
   ("-swarm", Arg.Unit (fun () -> opt_csmith_swarm := true),
    " randomly enable csmith features");
   ("-no_pointers", Arg.Unit (fun () -> opt_pointers := false),
    " disable pointers when generating programs using csmith");
   ("-seed", Arg.Int (fun i -> opt_csmith_seed := Some (i-1)),
    " supply a seed to csmith; with -repeat the seed is increased at each run");
   ("-max_seed", Arg.Int (fun i -> opt_csmith_max_seed := Some i),
    " max seed with -repeat");
   ("-compiler",
    Arg.String
      (fun x -> match x with
      | "icc" -> opt_compiler := Icc
      | "clang" -> opt_compiler := Clang
      | "gcc" -> opt_compiler := Gcc
      | _ -> error "unknown compiler"),
    " <clang|gcc|icc> specify compiler");
   ("-unopt",
    Arg.String (fun x -> opt_unopt := x),
    "options passed to the compiler for the unoptimised code");
   ("-opt",
    Arg.String (fun x -> opt_opt := x),
    "options passed to the compiler for the optimised code");
   ("-debug", Arg.Unit (fun () -> Analyse2.opt_debug := true),
    " print some debug informations");
   ("-debug_exec", Arg.Unit (fun () -> Util.opt_debug_exec := true),
    " dump the invocation of some external tools")
 ]

(* file generation via csmith *)

let csmith_bin =
  "csmith"

let coin =
  Random.self_init();
  fun () -> Random.bool()

let csmith_swarm_volatile () =
  let swarm_options = 
    [ ("--comma-operators","--no-comma-operators");
      ("--compound-assignment","--no-compound-assignment");
      ("--embedded-assigns","--no-embedded-assigns");
      ("--pre-incr-operator","--no-pre-incr-operator");
      ("--post-incr-operator","--no-post-incr-operator");
      ("--pre-decr-operator","--no-pre-decr-operator");
      ("--post-decr-operator","--no-post-decr-operator");
      ("--jumps","--no-jumps");
      ("--divs","--no-divs");
      ("--muls","--no-muls");
      ("--longlong","--no-longlong");
      ("--math64","--no-math64");
      ("--no-structs","--no-structs");
      ("--packed-struct","--no-packed-struct");
      ("--volatile-pointers","--no-volatile-pointers");
      ("--arrays","--no-arrays");
      ("--consts","--no-consts");
      ("--const-pointers","--no-const-pointers") ] in
  List.fold_left (fun s e -> s^" "^e) "" (List.map (fun (y,n) -> if coin() then y else n) swarm_options)

let csmith_opts () =
  "--max-expr-complexity "^(string_of_int !opt_max_expr_complexity)
  ^ " --max-funcs "^(string_of_int !opt_max_funcs)
  ^ " --no-checksum"
  ^ " --no-unions --no-argc "
  ^ ( if !opt_volatile 
    then " --volatiles --strict-volatile-rule --no-addr-taken-of-locals " ^ 
      (if !opt_csmith_swarm then csmith_swarm_volatile() else " --no-consts ")
    else " --no-volatiles --no-consts")
  ^ (match !opt_csmith_seed with
    | None -> " "
    | Some i -> " --seed "^(string_of_int i)^" ")
  ^ (if not !opt_structs then "--no-structs " else "--no-bitfields")
  ^ (if not !opt_arrays then "--no-arrays " else "")
  ^ (if not !opt_pointers then "--no-pointers " else "")

let csmith_opts_atomics () =
  "--max-expr-complexity "^(string_of_int !opt_max_expr_complexity)
  ^ " --max-funcs "^(string_of_int !opt_max_funcs)
  ^ " --no-pointers --no-unions --no-volatiles --no-arrays --no-structs"
  ^ " --no-consts --no-incr-decr-operators --no-checksum --no-comma-operators"
  ^ (if !opt_generate = (Some C) then " --atomics " else " --cpp-atomics ")

let csmith_opts_locks () =
  "--max-expr-complexity "^(string_of_int !opt_max_expr_complexity)
  ^ " --max-funcs "^(string_of_int !opt_max_funcs)
  ^ " --no-pointers --no-unions --no-volatiles --no-arrays --no-structs"
  ^ " --no-consts --locks --no-jumps --no-incr-decr-operators --no-checksum"

let csmith_include = "-I " ^ Config.find_config "csmith_include_dir"

let find_unused_name filetype =
  let rec find c : string =
    ( try
        let fd = Unix.openfile ((string_of_int c)^"_lock") [Unix.O_CREAT; Unix.O_EXCL] 0o644 in
        Unix.close fd;
        let name_c = (string_of_int c)^".c" in
        let name_cpp = (string_of_int c)^".cpp" in
        if Sys.file_exists name_c || Sys.file_exists name_cpp
        then (warning ("c or cpp file "^(string_of_int c)^" exists, but no lock found"); find (c+1))
        else match filetype with
          | "c" -> (string_of_int c)^".c"
          | "cpp" -> (string_of_int c)^".cpp"
          | _ -> error "unknown filetype"
      with Unix.Unix_error _ -> find (c+1) )
  in find 0

let icc_options filetype = match filetype with
  | "c"   -> "icc -Werror=uninitialized -fno-zero-initialized-in-bss -g -gdwarf-2 "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
  | "cpp" -> error "icc and c++ not implemented"
  | _     -> error "unknown filetype"

let gcc_options filetype = match filetype with
  | "c"   -> "gcc -Werror=uninitialized -fno-zero-initialized-in-bss -g -gdwarf-2 "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
(*  | "c"   -> "gcc --param allow-store-data-races=0 -Werror=uninitialized -fno-zero-initialized-in-bss -g -gdwarf-2 "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "") *)

  (* | "c"   -> "gcc -Werror=uninitialized -fno-zero-initialized-in-bss -g -gdwarf-2 "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "") *)

  | "cpp" -> "gcc --param allow-store-data-races=0 -fpermissive  -fno-zero-initialized-in-bss -g -std=c++11 -gdwarf-2 "^csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
  | _     -> error "unknown filetype"

let arm_gcc_options filetype =
  let csmith_include = "-I " ^ Config.find_config "csmith_include_dir" in
  match filetype with
  | "c"   -> Config.find_config "gcc_arm_bin" ^ " --param allow-store-data-races=0 -Werror=uninitialized -fno-zero-initialized-in-bss -g -gdwarf-2 "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
  | "cpp" -> error "cpp not yet supported for arm"
  | _     -> error "unknown filetype"

let clang_options filetype = match filetype with
  | "c"   -> "clang -fno-zero-initialized-in-bss -g "^ csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
  | "cpp" -> "clang -fpermissive  -fno-zero-initialized-in-bss -g -std=c++11 "^csmith_include ^ " " ^ (if !opt_quiet then "2>/dev/null " else "")
  | _     -> error "unknown filetype"

(* let test_termination f =  *)
(*   let filetype = if Filename.check_suffix f ".c" then "c" else "cpp" in  *)
(*   let gcc_with_options = gcc_options filetype in  *)
(*   exec (gcc_with_options^f^" -o "^f^"_unopt"); *)
(*   match Sys.command ("timeout 5 ./"^f^"_unopt") with *)
(*   | 0   -> exec("rm "^f^"_unopt"); true *)
(*   | 143 | 255(\*hack, should return 143*\)| 124(\*on beaune*\) -> exec("rm "^f^"_unopt"); false (\*exit status returned when command times out*\) *)
(*   | a   -> error ("execution returned "^ (string_of_int a) ^" exit status while testing for termination") *)

let generate_c_file filetype =
  let name = find_unused_name filetype in
  (* TODO : add filetype option in csmith *)
  let opts =
    match !opt_atomics, !opt_locks with
      | true, false -> csmith_opts_atomics ()
      | false, true -> csmith_opts_locks ()
      | false, false -> csmith_opts ()
      | true, true -> error "csmith does not yet support both atomics and locks" in
(*  print_endline ("*** " ^ opts); *)
  exec_fail (csmith_bin^" "^opts^" -o "^name);
  name

(*     let rec run_csmith () =  *)
(*       exec_fail (csmith_bin^" "^opts^" -o "^name); *)
(*       match test_termination name with *)
(*       | true  -> name *)
(*       | false -> print_endline "generated program timedout...generating new program"; *)
(*       run_csmith () *)
(*     in run_csmith () *)

(* Valgrind tracing *)

let valgrind_options =
  "--vex-iropt-level=0 --vex-guest-chase-thresh=0 --tool=interceptor --compact-mode=yes --get-values=yes"

let pin_options () =
  " -t " ^ Config.find_config "pin_so" ^ " -mark "
  ^ (if !opt_ir_unsound then "-unsound " else "")

(* dump the assembler *)
let asm compiler f =
  let fp = Filename.chop_extension f in
  let filetype = if Filename.check_suffix f ".c" then "c" else "cpp" in
  match !opt_trace_arch with
    | Some X86 ->
        ( match compiler with
          | Gcc ->
            let gcc_with_options = gcc_options filetype in
            (* assembly unopt & opt *)
            exec_fail (gcc_with_options ^ " " ^ !opt_unopt^" -S "^f^" -o "^fp^"_unopt.s");
            exec_fail (gcc_with_options ^ " " ^ !opt_opt^" -S "^f^" -o "^fp^"_opt.s")
          | Clang ->
            let clang_with_options = clang_options filetype in
            exec_fail (clang_with_options ^" " ^ !opt_unopt^" "^f^" -S -o "^fp^"_unopt.s");
            exec_fail (clang_with_options ^" " ^ !opt_opt^" "^f^" -S -o "^fp^"_opt.s")
            (* let clang_with_options = clang_options filetype in *)
            (* Util.split_main f; *)
            (* exec_fail (clang_with_options^" "^ !opt_unopt^" "^fp^"_body.c"^" -S -o "^fp^"_body_unopt.s"); *)
            (* exec_fail (clang_with_options^" "^ !opt_unopt^" "^fp^"_main.c"^" -S -o "^fp^"_main_unopt.s"); *)

            (* exec_fail (clang_with_options^" "^ !opt_opt^" "^fp^"_body.c"^" -S -o "^fp^"_body_opt.s"); *)
            (* exec_fail (clang_with_options^" "^ !opt_opt^" "^fp^"_main.c"^" -S -o "^fp^"_main_opt.s") *)

          | Icc ->
            let icc_with_options = icc_options filetype in
            exec_fail (icc_with_options ^ " " ^ !opt_unopt^" -S "^f^" -o "^fp^"_unopt.s");
            exec_fail (icc_with_options ^ " " ^ !opt_opt^" -S "^f^" -o "^fp^"_opt.s"))

    | Some ARM ->
      let arm_gcc_with_options = arm_gcc_options filetype in
      exec_fail (arm_gcc_with_options^" "^ !opt_unopt^" "^f^" -S -o "^fp^"_unopt.s");
      exec_fail (arm_gcc_with_options^" "^ !opt_opt^" "^f^" -S -o "^fp^"_opt.s")
    | None ->
      error "option -asm requires -trace <arch>"

(* compiles and traces *)

let raw_val_to_val dwarf vat st vr : value * symbol_table =
  try
    let vr = Z.to_int64 vr in
    let val_opt =
      let found = ref None in
      Hashtbl.iter (fun var_addr (var_name, var_size) ->
        if !found = None then
          if vr >= var_addr && (Int64.sub vr var_addr) < (Int64.of_int var_size) then
            found := Some (var_name, Int64.to_int var_addr, Int64.to_int (Int64.sub vr var_addr))
      ) vat;
      !found
    in
    match val_opt with
      | Some (var_name, base, offset) ->
        let st,vid = add_symbol_table st var_name in
        let type_tag = Dwarfparse.compute_type_tag dwarf (the (Dwarfparse.find_var_type_node dwarf var_name)) in
        let att,_,_ = Dwarfparse.resolve_address type_tag base (Int64.to_int vr) in
        Pointer (vid, offset, att), st
      | None -> NonPointer (Z.of_int64 vr), st
  with Z.Overflow ->
    NonPointer vr, st

let elaborate_raw_trace_pointers dwarf vat st (raw_trace:raw_trace) : event list * symbol_table =

  let get_atomic_load_attr index =
    match index with
      | 0 -> Seq_cst
      | 1 -> Acquire
      | 2 -> Relaxed
      | _ -> error ("gala: computed mod not in range") in

  let get_atomic_store_attr index =
    match index with
      | 0 -> Seq_cst
      | 1 -> Release
      | 2 -> Relaxed
      | _ -> error ("gasa: computed mod not in range") in

  let raw_loc_to_loc vat st rl : loc * symbol_table * int option* bool =
    try
      (* 1- find var_name, var_size, base and offset *)
      let var_name, var_size, base, offset =
        match
          let found = ref None in
          Hashtbl.iter (fun var_addr (var_name, var_size) ->
            if !found = None then
              if rl >= var_addr &&  (Int64.to_int (Int64.sub rl var_addr)) < var_size then
                found := Some (var_name, var_size, (Int64.to_int var_addr), Int64.to_int (Int64.sub rl var_addr))
          ) vat;
          !found
        with
          | Some (var_name, var_size, base, offset) -> var_name, var_size, base, offset
          | None -> error "internal: raw_loc_to_loc, can't find var_name"
      in
      let st,vid = add_symbol_table st var_name in
      (* 2- compute location attributes *)
      let var_type_node =
        match Dwarfparse.find_var_type_node dwarf var_name with
          | None -> warning (Printf.sprintf "cannot find a dwarf entry for %s" var_name); raise Not_found
          | Some vtn -> vtn
      in
      let type_tag = Dwarfparse.compute_type_tag dwarf var_type_node in
      let att,_,volatile = Dwarfparse.resolve_address type_tag base (Int64.to_int rl)
      in
      (* 3- compute the atomic attribute *)
      let aa =
        if var_name.[0]='a' && var_name.[1]='_' then
          Some (
            try (int_of_string (String.sub var_name 2 (String.length var_name - 2)) mod 3)
            with Failure _ -> error ("unable to get aa var_number from "^ var_name) )
        else None in
      (* (\* 4- is volatile *\) *)
      (* let volatile = Dwarfparse.is_volatile dwarf var_type_node in *)
      (* 5- return *)
      (vid,offset,att),st,aa,volatile
    with _ ->
      (* the source program is likely to perform an out-of-bound access to an array *)
      warning (Printf.sprintf "cannot convert raw_loc_to_loc: %Lx" rl);
      (0,0,[]),st,None, false
  in
  let elaborate_event st re : event * symbol_table  =
    match re with
    | RLoad (rl,rv,s) ->
        let l,st,aa,volatile = raw_loc_to_loc vat st rl in
        let v,st = raw_val_to_val dwarf vat st rv in
        ( match aa with
          | None ->
            if volatile then VLoad (l, v, s), st else Load (l, v, s), st
          | Some aa -> ALoad (get_atomic_load_attr aa, l, v, s), st )
    | RStore (rl,rv,s) ->
        let l,st,aa,volatile = raw_loc_to_loc vat st rl in
        let v,st = raw_val_to_val dwarf vat st rv in
        ( match aa with
          | None ->
            if volatile then VStore (l, v, s), st else Store (l, v, s), st
          | Some aa -> AStore (get_atomic_store_attr aa, l, v, s), st )
    | RFlush ->
        Flush, st
    | RLock rl ->
        let l,st,_,_ = raw_loc_to_loc vat st rl in
        Lock l, st
    | RUnlock rl ->
        let l,st,_,_ = raw_loc_to_loc vat st rl in
        Unlock l, st
  in
  let rec map_elaborate_event st rt t =
    match rt with
    | [] -> (List.rev t),st
    | re::rtt ->
        let e,st = elaborate_event st re in
        map_elaborate_event st rtt (e::t)
  in
  map_elaborate_event st raw_trace []

let initial_state arch dwarf vat st exc =
  ( match arch with
    | X86 -> exec_fail (Config.find_config "objdump_x86_bin" ^ " -d -z -j .data -j .rodata -j .bss "^exc^" > "^exc^".dump")
    | ARM -> exec_fail (Config.find_config "objdump_arm_bin" ^ " -d -z -j .data -j .rodata -j .bss "^exc^" > "^exc^".dump"));
  let fd =
    try open_in (exc^".dump")
    with Sys_error _ -> error ("initial_state: can't open dump.") in

  let rec read_bits_init fd buf =
    let return_buf buf = String.concat "" (List.rev buf) in
    try
      let line = input_line fd in
      if String.length line = 0 || line.[0] = '\n' then return_buf buf else
        let sc_idx = String.index line ':' in
        let init_value_str =
          match arch with
            | X86 -> String.sub line (sc_idx+2) 48
            | ARM ->
              (* working around the idiosynchrasies of the arm-elf-objdump txt format *)
              let max_rl =
                try (Str.search_forward (Str.regexp (Str.quote ".word")) line 0) - (sc_idx+2)
                with Not_found -> 36 in
              let rl = String.sub line (sc_idx+2) max_rl in
              let rl' = Str.split (Str.regexp (Str.quote " ")) rl in
              let rl'' = String.concat "" rl' in
              let ol = String.make 49 ' ' in
              for i = 0 to (String.length rl'')-1 do
                ol.[i+(i/2)] <- rl''.[i];
              done;
              (String.trim ol)^" "
        in
        read_bits_init fd (init_value_str::buf)
    with End_of_file -> return_buf buf in

  let extract_n_bytes n hex =
    (* hex is a string of the form 00 00 00 00 ..*)
    let rec aux n s buf =
      if n = 0 then List.rev buf, try String.sub hex s ((String.length hex)-s) with _ -> ""
      else let byte = String.sub hex s 2 in
           aux (n-1) (s+3) (byte::buf)
    in aux n 0 []
  in

  let rec compute_init_evts var_id type_tag offset hex : event list * string * int =
    let debug s = if false then print_endline s in

    let arch_rev arch l = match arch with X86 -> List.rev l | ARM -> l in


    debug ("TYPE_TAG = "^(Dwarfparse.dump_type_tag type_tag));
    match type_tag with
      | Dwarfparse.TBase s ->
        debug ("TBase "^(string_of_int s));
        let bytes,hex_tail = extract_n_bytes s hex in
        let raw_value = Z.of_string ("0x"^(String.concat "" (arch_rev arch bytes))) in
        let (value,_) = raw_val_to_val dwarf vat st raw_value in
        let att,_,_ = Dwarfparse.resolve_address type_tag 0 offset in

        [ Init ((var_id, offset, att), value,s) ], hex_tail, (offset+s)

      | Dwarfparse.TArr (sl,att) ->
        debug ("TArr ");
        let array_size = List.hd sl in
        let elem_size = last sl in
        let num_elem = array_size / elem_size in
        (* extract num_elem times elem_size bytes, and recursively invoke cie *)
        let rec array_aux n offset hex : event list * string * int =
          if n = 0 then ([], hex, offset)
          else begin
            let el_evts, hex_tail, offset_tail = compute_init_evts var_id att offset hex in
            assert (offset_tail = offset+elem_size);
            let tail_evts, hex_tail, offset_tail = array_aux (n-1) offset_tail hex_tail in
            el_evts@tail_evts, hex_tail, offset_tail
          end
        in
        array_aux num_elem offset hex

      | Dwarfparse.TStr (l,s) ->
        debug ("TStr; s="^(string_of_int s) ^" offset=" ^(string_of_int offset)^" : " ^hex);

        let rec aux_str offset_str l hex =
          debug ("   aux_str: offset_str = "^(string_of_int offset_str));
          match l with
            | [] ->
              debug "empty";
              if s > offset_str
              then begin
                debug "padding at the end of the structure";
                let bytes,hex_tail = extract_n_bytes (s-offset_str) hex in
                let raw_value = Z.of_string ("0x"^(String.concat "" (arch_rev arch bytes))) in
                let (value,_) = raw_val_to_val dwarf vat st raw_value in
                [Init ((var_id,offset+offset_str,[Pad]), value , s-offset_str)], hex_tail, offset+s
              end else begin
                debug "no padding at the end of the structure";
                [], hex, offset+s
              end

            | (f,o,tt)::t ->
              let pad_evts,hex =
                if  o > offset_str then begin
                  debug ("internal padding : "^(string_of_int (o-offset_str)));
                  let padding_bytes,hex = extract_n_bytes (o-offset_str) hex in
                  let raw_value = Z.of_string ("0x"^(String.concat "" (arch_rev arch padding_bytes))) in
                  let (value,_) = raw_val_to_val dwarf vat st raw_value in
                  [Init ((var_id,offset+offset_str,[Pad]), value , o-offset_str)], hex
                end else [],hex in
              (* then extract the bits *)
              debug (" el, o: "^(string_of_int o) ^" remaining hex: "^hex);
              let ievts, hex_tail, hex_offset = compute_init_evts var_id tt (offset+o) hex in
              (* debug (String.concat "  ---  " (List.map (string_of_event st) ievts)); *)
              let event_size = Dwarfparse.size_of_type_tag tt in
              let tevts, hex_tail, hex_offset = aux_str (o+event_size) t hex_tail in
              pad_evts @ ievts @ tevts, hex_tail, hex_offset in

        aux_str 0 l hex

      | Dwarfparse.TUni s ->
        debug ("TUni "^(string_of_int s));
        let bytes,hex_tail = extract_n_bytes s hex in
        let raw_value = Z.of_string ("0x"^(String.concat "" (arch_rev arch bytes))) in
        let (value,_) = raw_val_to_val dwarf vat st raw_value in
        let att,_,_ = Dwarfparse.resolve_address type_tag 0 offset in

        [ Init ((var_id,offset,att), value,s) ], hex_tail, (offset+s)

      | Dwarfparse.TVolatile tt -> compute_init_evts var_id tt offset hex

  in



(*      let rec aux_str off l lev hex = *)
(*           debug ("TStr-aux  lev:" ^(string_of_int lev)^ " g:"^(string_of_int ((s-(off+lev)))) ^" : " ^hex); *)
(*        match l with *)
(*          | [] ->  *)
(*               debug "empty"; *)
(*               if s - (off+lev) > 0  *)
(*               then ( *)
(*                 debug "pad"; *)
(*              let _,hex = extract_n_bytes lev hex in  *)
(*                 let bits,rest = extract_n_bytes (s-(off+lev)) hex in *)
(*                 let raw_value = Int64.of_string ("0x"^(String.concat "" (List.rev bits))) in *)
(*                 let (value,_) = raw_val_to_val dwarf vat st raw_value in *)
(*                 let loc_att = [Pad] in  *)
(*                 [Init ((var_id,off+lev,loc_att), value , s-(off+lev))] ) *)
(*               else (debug "nopad";[]) *)

(*          | (f,o,tt)::t ->  *)
(*               debug ("TStr-aux  full, lev:" ^(string_of_int lev)^ " o-off: "^(string_of_int (o-off)) ^" : " ^hex); *)
(*               debug ("              , s:" ^(string_of_int s)^ " o: "^(string_of_int o)); *)
(*               (\* first skip the last event *\) *)
(*            let _,hex = extract_n_bytes lev hex in  *)
(*               (\* if o-lev-off is not 0, then it is padding *\) *)
(*            let pad_evts,hex =  *)
(*                 if  o-off-lev > 0 then begin *)
(*                   debug ("internal padding : "^(string_of_int (o-off-lev))); *)
(*                   let psize = o-off-lev in *)
(*                   let padding_bits,hex = extract_n_bytes psize hex in *)
(*                   let raw_value = Int64.of_string ("0x"^(String.concat "" (List.rev padding_bits))) in *)
(*                   let (value,_) = raw_val_to_val dwarf vat st raw_value in *)
(*                   let loc_att = [Pad] in  *)
(*                   [Init ((var_id,off+o+lev,loc_att), value , o-(off+lev))], hex *)
(*                 end else [],hex in *)
(*            (\* then extract the bits *\) *)
(*               debug (" remaining hex: "^hex); *)
(*               let event_size = Dwarfparse.size_of_type_tag tt in *)
(*            let bits,rest = extract_n_bytes event_size hex in *)
(*            let bits = String.concat " " bits in *)
(*               let ievts = compute_init_evts var_id vtt (offset+o) tt bits in *)
(* (\*              debug (String.concat "  ---  " (List.map (string_of_event st) ievts)); *\) *)
(*            pad_evts @ ievts @ (aux_str o t event_size hex) in *)
(*      aux_str 0 l 0 hex *)

(*       | _ -> [] *)



  let rec parse_dump st init_evts : symbol_table * event list =
    try
      let line = input_line fd in
      try (* ahem, how not to parse a file *)
        let rgx = Str.regexp "<.+>" in
        let _ = Str.search_forward rgx line 0 in
        let mtc = Str.matched_string line in
        let var_name = String.sub mtc 1 ((String.length mtc)-2) in
        if var_name.[0] = 'g'  || var_name.[0] = 'c' || var_name.[0] = 'a' then begin
          (* 1- analyse the type *)
          match Dwarfparse.find_var_type_node dwarf var_name with
            | Some ptr ->
                (* 2- read the bits *)
                let hex = read_bits_init fd [] in
                let type_tag = Dwarfparse.compute_type_tag dwarf ptr in
                let st,var_id = add_symbol_table st var_name in
                (* 3- analyse the bits according to the type *)
                let new_ie,_,_ = compute_init_evts var_id type_tag 0 hex in
                parse_dump st (new_ie@init_evts)
            | None -> parse_dump st init_evts
        end
        else parse_dump st init_evts
      with Not_found -> parse_dump st init_evts
    with End_of_file -> close_in fd; (st,init_evts)
  in
  let st,init_evts = parse_dump st [] in
  Sys.remove (exc^".dump");
  (init_evts,st)

let elaborate_raw_trace arch src st exc log =
  match Raw_trace_parse.parse_log log with
    | Some (raw_trace, ireads, ldlist, copts) ->
      (* Types.dump_raw_trace raw_trace; *)
      let dwarf = Dwarfparse.parse_dwarf arch exc in
      let vat = Modify_pintrace.construct_var_address_table arch exc in
      let t,st = elaborate_raw_trace_pointers dwarf vat st raw_trace in
      let init_evts,st = initial_state arch dwarf vat st exc in
      (* Types.dump_trace (t,st); *)
      let trace = {
        source     = src;
        executable = exc;
        events     = init_evts@t;
        ir_reads   = ireads;
        load_deps  = ldlist;
        comp_opts  = copts;
      } in
      Some (trace,st,dwarf)

    | None ->
      (* warning ("cannot raw_parse_trace "^log); *)
      None

let trace do_opt f instrumenting_tool compiler arch =
  let unopt_comp_string = ref "" in
  let opt_comp_string = ref "" in

  (* tracing function *)
  let trace_tool st exc =
    match arch with
      | X86 ->
        ( match instrumenting_tool with
          | Valgrind ->
            let rv =
              exec_timeout 5 ("valgrind "^valgrind_options^" --log-file="^exc^".log ./"^exc) in
            ( match rv with
              | None -> None
              | Some v -> elaborate_raw_trace X86 f st exc (exc^".log"))

          | Pin      ->
            let rv = exec_timeout 5 ("pin "^pin_options ()^" -o "^exc^".log -- ./"^exc ^" > /dev/null") in
            if !opt_ir_replay then exec_fail ("cp "^exc^".log "^exc^".raw_log");
            (* TODO restore the Copts infos in the traces? *)
            (* ignore (Sys.command ("echo \"\n*Copts:"^ !unopt_comp_string^"\" >> "^exc^".log")); *)
            match rv with
              | None -> None
              | Some v -> elaborate_raw_trace X86 f st exc (exc^".log"))

              (* (match rv with Some _ -> Modify_pintrace.modify (b^".log") ("./"^b) | None -> ()); *)
              (* exec_fail ("modify_pintrace -pinlog "^b^".log -exec ./"^b) | None -> ()); *)
      | ARM ->
        ( let rv = exec_timeout 5 ("arm_tracer "^exc) in
          match rv with
          | Some r -> elaborate_raw_trace ARM f st exc (exc^".log")
          | None -> None ) in

  let fp =
    (Filename.chop_extension f)
    ^ if do_opt then "" else match compiler with Gcc -> "_gcc" | Clang -> "_clang" | Icc -> "_icc" in
  let filetype =
    if Filename.check_suffix f ".c" then "c" else "cpp" in

  (* compile unopt & opt *)
  (match arch with
    | X86 ->
      ( match compiler with
        | Gcc  ->
          let gcc_with_options = gcc_options filetype in
          unopt_comp_string := (gcc_with_options^" "^ !opt_unopt^" "^f^" -o "^fp^"_unopt");
          exec_fail !unopt_comp_string;
      (* exec_fail ("clang -g -fno-zero-initialized-in-bss "^csmith_include^" "^f^" -o "^fp^"_unopt"); *)
          if do_opt then (
            opt_comp_string := (gcc_with_options^" "^ !opt_opt^" "^f^" -o "^fp^"_opt");
            exec_fail !opt_comp_string )
        | Clang ->
          let clang_with_options = clang_options filetype in
          unopt_comp_string := (clang_with_options^" "^ !opt_unopt^" "^f^" -o "^fp^"_unopt");
          exec_fail !unopt_comp_string;
          if do_opt then (
            opt_comp_string := (clang_with_options^" "^ !opt_opt^" "^f^" -o "^fp^"_opt");
            exec_fail !opt_comp_string )
          (* let clang_with_options = clang_options filetype in *)
          (* Util.split_main f; *)
          (* unopt_comp_string := (clang_with_options^" "^ !opt_unopt^" "^fp^"_body.c"^" -c -o "^fp^"_body_unopt.o"); *)
          (* exec_fail !unopt_comp_string; *)
          (* unopt_comp_string := (clang_with_options^" "^ !opt_unopt^" "^fp^"_body_unopt.o "^fp^"_main.c"^" -o "^fp^"_unopt"); *)
          (* exec_fail !unopt_comp_string; *)

          (* if do_opt then ( *)
          (*   opt_comp_string := (clang_with_options^" "^ !opt_opt^" "^fp^"_body.c"^" -c -o "^fp^"_body_opt.o"); *)
          (*   exec_fail !opt_comp_string; *)
          (*   opt_comp_string := (clang_with_options^" "^ !opt_opt^" "^fp^"_body_opt.o "^fp^"_main.c"^" -o "^fp^"_opt"); *)
          (*   exec_fail !opt_comp_string; ) *)


        | Icc ->
          let icc_with_options = icc_options filetype in
          Util.split_main f;
          unopt_comp_string := (icc_with_options^" "^ !opt_unopt^" "^fp^"_body.c"^" -c -o "^fp^"_body_unopt.o");
          exec_fail !unopt_comp_string;
          unopt_comp_string := (icc_with_options^" "^ !opt_unopt^" "^fp^"_body_unopt.o "^fp^"_main.c"^" -o "^fp^"_unopt");
          exec_fail !unopt_comp_string;

          if do_opt then (
            opt_comp_string := (icc_with_options^" "^ !opt_opt^" "^fp^"_body.c"^" -c -o "^fp^"_body_opt.o");
            exec_fail !opt_comp_string;
            opt_comp_string := (icc_with_options^" "^ !opt_opt^" "^fp^"_body_opt.o "^fp^"_main.c"^" -o "^fp^"_opt");
            exec_fail !opt_comp_string; ) )

    | ARM ->
        let arm_gcc_with_options = arm_gcc_options filetype in
        unopt_comp_string := (arm_gcc_with_options^" "^ !opt_unopt^" "^f^" -o "^fp^"_unopt");
          exec_fail !unopt_comp_string;
        if do_opt then (
            opt_comp_string := (arm_gcc_with_options^" "^ !opt_opt^" "^f^" -o "^fp^"_opt");
            exec_fail !opt_comp_string )
  );

  (* trace unopt & opt *)

  let st = empty_symbol_table in

  (* print_endline "unopt trace"; *)
  let unopt_trace_opt = trace_tool st (fp^"_unopt") in

  match unopt_trace_opt with
    | None ->
      if !opt_real_quiet then print_char '_' else warning ("timeout while tracing: "^fp^"_unopt");
      List.iter
        (fun x -> if Sys.file_exists x then Sys.remove x)
        ((if !opt_repeat || !opt_repeat_forever then [f] else [])
         @ [fp^"_unopt.log"; fp^"_unopt"; fp^"_opt"]
         @ (if !opt_repeat || !opt_repeat_forever then [fp^"_lock"] else []));
      None

    | Some (unopt_trace,st,dwarf_unopt) -> begin
      (* print_endline "opt trace"; *)
      let opt_trace_opt = trace_tool st (fp^"_opt") in
      match opt_trace_opt with
        | None ->
          if !opt_real_quiet then print_char '_' else warning ("timeout while tracing: "^fp^"_opt");
          List.iter
            (fun x -> if Sys.file_exists x then Sys.remove x)
            ((if !opt_repeat || !opt_repeat_forever then [f] else [])
             @ [fp^"_unopt.log"; fp^"_unopt"; fp^"_opt.log"; fp^"_opt"]
             @ (if !opt_repeat || !opt_repeat_forever then [fp^"_lock"] else []));
          None
        | Some (opt_trace,st,dwarf_opt) ->
          Some ({unopt_trace with comp_opts = !unopt_comp_string },
                {opt_trace with comp_opts = !opt_comp_string }, st, dwarf_unopt)
    end


(*     (\* abort if tracing unopt did not return in 30 seconds *\) *)
(*     if do_opt then ignore (tool (fp^"_opt")); *)
(*     (\*construct the var_size_table*\) *)
(* (\*     let size_table = initial_var_size (fp^"_unopt") arch in *\) *)
(* (\* (\\*    let size_type_table = initial_var_type_2 f size_table in *\\) *\) *)
(* (\*     let size_type_table = initial_var_type (fp^"_unopt") size_table in  *\) *)

(* (\*     (\\* get the initial state *\\) *\) *)
(* (\*     initial_state (fp^"_unopt") size_type_table arch; *\) *)
(* (\*     if do_opt then initial_state (fp^"_opt") size_type_table arch; *\) *)
(* (\*     (\\*construct var_size_table*\\) *\) *)
(* (\*     (\\* merge initial state and trace *\\) *\) *)
(* (\*     exec_fail ("cat "^fp^"_unopt.log >> "^fp^"_unopt_init.log"); *\) *)
(* (\*     Sys.rename (fp^"_unopt_init.log") (fp^"_unopt.log"); *\) *)
(*     if do_opt then begin *)
(*       (\* exec_fail ("cat "^fp^"_opt.log >> "^fp^"_opt_init.log"); *\) *)
(*       (\* Sys.rename (fp^"_opt_init.log") (fp^"_opt.log"); *\) *)
(* (\*      ignore (Sys.command ("echo \"\n*Copts:"^ !opt_comp_string^"\" >> "^fp^"_opt.log")); *\) *)
(*     end; *)
(*     (\* add the new traces to the list of log files *\) *)
(*     (\* FIXME: check if should do this when tracing for delta *\) *)
(*     if do_opt then log_files := ((fp^"_unopt.log", fp^"_opt.log"),Some filetype) :: !log_files; *)
(*     true *)
(*   end else begin *)
(*     if !opt_real_quiet then print_char '_' else warning ("\ntimeout while tracing: "^fp^"_unopt"); *)
(*     List.iter  *)
(*       (fun x -> if Sys.file_exists x then Sys.remove x) *)
(*       [f; fp^"_unopt.log"; fp^"_unopt"; fp^"_opt"; fp^"_lock"]; *)
(*     false *)
(*   end *)

let gen_env_vars () =
  let analyse =
    match !opt_analyse with
    | true   -> "-analyse"
    | false  -> ""
  and analyse_volatile_only =
    match !opt_analyse_volatile_only with
    | true   -> "-analyse_volatile_only"
    | false  -> ""
  and analyse_volatile_cached =
    match !opt_analyse_volatile_cached with
    | true   -> "-analyse_volatile_cached"
    | false  -> ""
  and analyse_count_only =
    if !opt_analyse_count_only then "-analyse_count_only" else ""
  and trace =
    match !opt_trace_arch with
      | Some X86 -> "x86"
      | Some ARM -> "arm"
      | None -> error "test_case reduction requires -trace"
  and compiler =
    match !opt_compiler with
    | Icc   -> "icc"
    | Gcc   -> "gcc"
    | Clang -> "clang"
  and unsound =
    match !opt_ir_unsound with
    | true      -> ""
    | false     -> "-no_unsound_ir"
  and ir_replay =
    match !opt_ir_replay with
    | true      -> "-ir_replay"
    | false     -> ""
  and dumptraces =
    match !opt_dump_traces with
    | All -> "-dump_traces_init"
    | NoInit -> "-dump_traces"
    | Quiet -> ""
  and timeout =
    match !opt_timeout with
    | None -> "3600"  (* one hour max per test-case reduction iteration...  basically it won't converge *)
    | Some i -> string_of_int i
  and timeout_delta =
    if !opt_timeout_delta then "-timeout_delta" else ""
  and comp_opts =
    ( if !opt_unopt = ""
      then ""
      else "-unopt "^ !opt_unopt )
    ^ "-opt "^ !opt_opt
  and csmith_include = Config.find_config "csmith_include_dir"
  in
 " TRACE="^trace
  ^ " UNSOUND="^unsound
  ^ " IRREPLAY="^ir_replay
  ^ " ANALYSE="^analyse
  ^ " ANALYSE_VOLATILE_ONLY="^analyse_volatile_only
  ^ " ANALYSE_VOLATILE_CACHED="^analyse_volatile_cached
  ^ " ANALYSE_COUNT_ONLY="^analyse_count_only
  ^ " COMPILER="^compiler
  ^ " COMPOPTS=\""^comp_opts^"\""
  ^ " DUMPTRACES="^dumptraces
  ^ " TIMEOUT="^timeout
  ^ " TIMEOUT_DELTA="^timeout_delta
  ^ " CSMITHINCLUDE="^csmith_include

(* analysis *)

let run_analysis timeout_delta delta (unopt_trace,opt_trace,st,dwarf) =

  let filetype = if Filename.check_suffix unopt_trace.source "c" then "c" else "cpp" in

  (* let filetype = match file with *)
  (* | Some a -> a *)
  (* | None   ->  *)
  (*     if delta != None || !opt_clean then warning  *)
  (*     "unable to run delta/clean because undetermined original filetype (passed as a log file?)"; "" in *)


  let an_function () =
    (* if !opt_analyse_count_only then  *)  (* FIXME *)
    (*   fun () -> Some (Analyse2.count_only (unoptlog,optlog)) *)
    (* else if !opt_analyse_stat then  *)
    (*   fun () ->  *)
    (*  (\* let (_,u) = Analyse2.record_opt_stats unoptlog None  *)
    (*     in  Some (fst (Analyse2.record_opt_stats optlog (Some u))) *\) *)
    (*  Some (fst (Analyse2.record_opt_stats unoptlog None)) *)
    (* else *)
    (*   fun () ->  *)
    Analyse2.analyse
      !opt_ir_replay !opt_timeout !opt_dump_traces !opt_real_quiet 
      !opt_analyse_volatile_only !opt_analyse_volatile_cached 
      (unopt_trace,opt_trace,st,dwarf)  in

  let an_result = an_function () in

  let do_delta () = match delta with
    | None -> ()
    | Some d -> if filetype == "" then () else begin
(*      let tmp_name = unoptlog) *)
        let fp = Filename.basename (Filename.chop_extension unopt_trace.source) in
         let reduction_success =
           ( match d with
             | Delta ->
               print_endline ("running delta on: "^fp);
               exec_fail ("env "^ gen_env_vars ()
                          ^" delta -test=`which delta_check.sh` -suffix=."
                          ^filetype^" -cp_minimal="^fp^"-min."^filetype^" "
                          ^fp^"."^filetype^" > /dev/null");
               true
             | CReduce ->
             (* create a new directory, copy the source as input.c, and invoke creduce *)
               print_endline ("running creduce on: "^fp);
               let dirname = fp^"_dir" in
             (* FIXME: proper dir error check *)
               ignore (Sys.command ("rm -rf "^dirname));
               Unix.mkdir dirname 0o700;
               Unix.chdir dirname;
               ignore (Sys.command ("cp ../"^fp^"."^filetype^" input."^filetype));
               if (String.compare filetype "c") = 0
               then
                 ignore (Sys.command "cp `which creduce_test_c.sh` ./creduce_test.sh")
               else
                 ignore (Sys.command "cp `which creduce_test_cpp.sh` ./creduce_test.sh");
               (* let cr_st = exec ("env "^gen_env_vars ()
		                          ^" c_reduce.pl ./creduce_test.sh input."^filetype
                                          ^ if !opt_quiet then "  > /dev/null" else "" ) in *)
               let cr_st = exec ("env "^gen_env_vars ()
				 ^" creduce -n 1 creduce_test.sh input."^filetype
				 ^ if !opt_quiet then " > /dev/null" else "" ) in
               if cr_st = 0 then Sys.rename ("input."^filetype) ("../"^fp^"-min."^filetype);
               Unix.chdir "..";
               if cr_st = 0 then true else false ) in
         if reduction_success then begin
           let final_trace = trace true (fp^"-min."^filetype) !opt_tool !opt_compiler
             (match !opt_trace_arch with Some x -> x | None -> error "internal: cannot happen in do_delta") in
           match final_trace with
             | None ->
                 warning ("timeout while tracing a file generated by test-case reduction")
             | Some (unopt_trace,opt_trace,st,dwarf) ->
                 ignore (Analyse2.analyse !opt_ir_replay !opt_timeout All false !opt_analyse_volatile_only !opt_analyse_volatile_cached
                           (unopt_trace,opt_trace,st,dwarf))
         end
    end
  in

  let clean () =
    if !opt_clean then
      let tmp_name = unopt_trace.executable in
      let fp =
        try String.sub tmp_name 0 ((String.length tmp_name)-6)
        with _ -> "" in
      List.iter
        (fun x -> if Sys.file_exists x then Sys.remove x)
        [fp^".c"; fp^".cpp"; fp^"_unopt.log"; fp^"_opt.log"; fp^"_unopt.raw_log"; fp^"_opt.raw_log"; fp^"_lock"]
  in

  match an_result with
    | None ->
      if timeout_delta then begin
        print_endline "timeout_delta";
        do_delta ();
        true
      end else begin
        (* print_endline "TIMEOUT";  *)
        clean();
        false
      end
    | Some true ->
      clean ();
      false
    | Some false ->
      do_delta ();
      true

(* baseline *)

let rec compare_baseline delta dump_traces f : bool = false
  (* let fp = (Filename.chop_extension f) in *)
  (* if trace false f !opt_tool Gcc X86 then begin *)
  (*   ignore (trace false f !opt_tool Clang X86); *)
  (*   let result =  *)
  (*     print_endline "gcc vs clang: "; *)
  (*     let an_result_gcc_clang =  *)
  (*    Analyse2.analyse false !opt_timeout dump_traces !opt_analyse_write_only false (fp^"_gcc_unopt.log", fp^"_clang_unopt.log") in *)
  (*     if an_result_gcc_clang = Some true then begin *)
  (*    print_endline "clang vs gcc: "; *)
  (*    let an_result_clang_gcc = *)
  (*      Analyse2.analyse false !opt_timeout dump_traces !opt_analyse_write_only false (fp^"_clang_unopt.log", fp^"_gcc_unopt.log") in *)
  (*    match an_result_gcc_clang, an_result_clang_gcc with *)
  (*      | Some true, Some true -> true *)
  (*      | _, _ -> false *)
  (*     end else false *)
  (*   in *)
  (*   if !opt_clean then *)
  (*     List.iter  *)
  (*    (fun x -> if Sys.file_exists x then Sys.remove x) *)
  (*    [f; fp^"_gcc_unopt.log"; fp^"_clang_unopt.log"; fp^"_lock"]; *)
  (*   result *)

  (*   (\* let is_delta, delta_command = match delta with *\) *)
  (*   (\*   | Some Delta -> true, "delta" *\) *)
  (*   (\*   | Some CReduce -> true, "c_reduce" *\) *)
  (*   (\*   | None -> false, "" *\) *)
  (*   (\* in  *\) *)
  (*   (\* if is_delta && (not an_result_clang_gcc || not an_result_gcc_clang)  *\) *)
  (*   (\* then ( (\\*FIXME: see run_analysis.*\\) *\) *)
  (*   (\*   exec_fail  *\) *)
  (*   (\*      ("env "^ gen_env_vars () ^" "^delta_command^" -test=`which delta_check.sh`" *\) *)
  (*   (\*        ^" -suffix=.c -cp_minimal="^fp^"-min.c "^fp^".c"); *\) *)
  (*   (\* (\\*exec ("delta -test=`which delta_check_baseline.sh` -suffix=.c -cp_minimal="^fp^"-min.c "^fp^".c");*\\) *\) *)
  (*   (\*   compare_baseline None All (fp^"-min.c")) *\) *)
  (* end else begin (\* timeout while tracing *\)  *)
  (*   if !opt_clean then *)
  (*     List.iter  *)
  (*    (fun x -> if Sys.file_exists x then Sys.remove x) *)
  (*    [f; fp^"_clang_unopt"; fp^"_gcc_unopt.log"; fp^"_clang_unopt.log"; fp^"_lock"]; *)
  (*   true *)
  (* end *)

(* main *)

let _ =
  let c_files = ref [] in
  Arg.parse opts (fun s -> c_files := s::!c_files)
    "usage: cmmtest [options] <file1> .. <filen>";

  (* 1- sort the .c from the .log files *)
  (* c_files :=  *)
  (*     List.filter (fun fn -> Filename.check_suffix fn ".c" || Filename.check_suffix fn ".cpp" ) !tmp_log_files; *)
  (* log_files :=               *)
  (*   combine  *)
  (*   (List.filter (fun fn -> Filename.check_suffix fn ".log") !tmp_log_files) *)
  (*   []; *)

  (* 2- error checking of the options *)
  ( match !opt_generate, !opt_baseline, !opt_trace_arch, !opt_asm, !opt_dump_traces, (!opt_analyse || !opt_analyse_volatile_only || !opt_analyse_volatile_cached), !opt_delta, !opt_repeat with
    | None, _, _, _, _, _, _, true -> error "-repeat enabled only with -generate"
    | Some _ , _, _, _, _, _, _, true  when (List.length !c_files > 0) -> error "additional files not allowed with -repeat"
    | _, true, _, _, _, true, _, _ -> error "use either -baseline or -analyse"
    | _, false, _, _, _, false, Some _, _ -> error "-delta requires -analyse or -baseline"
    | _, false, _, _, All, false, _, _ -> error "-dump_traces_init requires -analyse or -baseline"
    | _, false, _, _, NoInit, false, _, _ -> error "-dump_traces requires -analyse or -baseline"
    | _, false, None, false, _, false, _, _ when (List.length !c_files > 0) -> error "don't know what to do with the specified files"
    (*TODO: assert -repeat to be used only with -generate, and no c/log files*)
    | _ -> () );

  (* 3- process -generate *)
  let rec repeat_testing () =
    (* ugly way to deal with -seed and -repeat *)
    ( match !opt_csmith_seed, !opt_csmith_max_seed with 
      | Some i, Some m -> 
        opt_csmith_seed := Some (i+1); 
        if i = m then exit 0;
        print_endline ("\n *** SEED : "^(string_of_int (i+1)) ^" ***\n")
      | Some i, None -> 
        opt_csmith_seed := Some (i+1);
        print_endline ("\n *** SEED : "^(string_of_int (i+1)) ^" ***\n")
      | None, _ -> () );

    (match !opt_generate with
    | Some C ->
        let name = generate_c_file "c" in
        if !opt_repeat then c_files := [name]
        else c_files := name :: !c_files
    | Some Cpp ->
        let name = generate_c_file "cpp" in
        if !opt_repeat then c_files := [name]
        else c_files := name :: !c_files
    | None -> ());

    (* 4- process -asm *)
    if !opt_asm then begin
      List.iter (asm !opt_compiler) !c_files
    end;

    (* 5- process -trace *)
    let trace_list =
      ((* if not !opt_baseline then *) match !opt_trace_arch with
        | Some arch ->
          option_map (fun x -> trace true x !opt_tool !opt_compiler arch) !c_files
        | None -> []) in

    (* 6- process -analyse *)
    if !opt_analyse || !opt_analyse_volatile_only || !opt_analyse_volatile_only
                    || !opt_analyse_count_only || !opt_analyse_stat then begin
      if !opt_repeat then begin
        match trace_list with
          | [] -> (* timeout during tracing *) repeat_testing ()
          | [ t ] ->
            ( match run_analysis !opt_timeout_delta !opt_delta t with
              | false ->
                (* if !opt_analyse_stat && !Types.no_tests > 0 then begin *)
                (*   Types.no_tests := !Types.no_tests-1; *)
                (*   repeat_testing () *)
                (* end else begin *)
                (*   if !opt_analyse_stat && !Types.no_tests = 0 then begin *)
                (*     Analyse2.stats_process (); *)
                (*     exit 0; *)
                (*   end *)
                (*   else  *)
                repeat_testing ()

              | true  -> if !opt_repeat_forever then repeat_testing () else () )
          | _ :: _ -> error "internal: opt_repeat with list > 1"
      end else
        List.iter (fun x -> ignore (run_analysis !opt_timeout_delta !opt_delta x)) trace_list
      end

    (* 7- process -baseline *)
    (* if !opt_baseline then begin *)
    (*   if !opt_repeat then begin *)
    (*  match !c_files with *)
    (*    | [] -> repeat_testing () *)
    (*    | _ -> *)
    (*         match compare_baseline !opt_delta !opt_dump_traces (List.hd !c_files) with  *)
    (*           | true -> c_files:= []; repeat_testing () *)
    (*           | false -> ()  *)
    (*   end else *)
    (*   (\* for each c-file, trace it with gcc and clang, and run the analyser in both directions *\) *)
    (*  List.iter (fun x -> ignore (compare_baseline !opt_delta !opt_dump_traces x)) !c_files *)
    (* end *)

  in
    repeat_testing ();



(*** ATTIC ***)

(*      matchTraces !log_files *)
(*  | _ -> () *)
      (* if String.compare !trace_file_unopt "" == 0  *)
      (*    || String.compare !trace_file_opt "" == 0 *)
      (* then error "must specify -unopt && -opt options" *)
      (* else match !mode with  *)
      (* | Elimination ->  *)
      (*          Compare.compare !trace_file_unopt !trace_file_opt *)
      (* | Reorder -> () ) *)
          (*compute reordering set*)
          (* let rset = [(1,3);(4,8);(4,6)] in *)
          (* check_reorder !trace_file_unopt !trace_file_opt rset) *)


(* let type_to_size t = *)
(*   match t with  *)
(*     | "long" -> 8 *)
(*     | "uint64_t" | "int64_t" -> 8 *)
(*     | "uint32_t" | "int32_t" -> 4 *)
(*     | "uint16_t" | "int16_t" -> 2 *)
(*     | "uint8_t" | "int8_t" -> 1 *)
(*     | _ -> error ("type_to_size, unknown type: "^t) *)

(* let array_to_size sl = *)
(*   match sl with *)
(*     | Some sl -> *)
(*       List.fold_left (fun x y -> x*y) 1 *)
(*      (List.map (fun s -> try int_of_string s with _ -> 1) sl) *)
(*     | None -> 1 *)



(* let initial_var_type_2 source size_table =  *)
(*   let debug s = if false then print_endline s in *)
(*   (\* this is a fragile, ad-hoc, parser for type declarations in C files *\) *)
(*   let var_type_table = Hashtbl.create 15 in *)
(*   let fd = *)
(*     try open_in source  *)
(*     with _ -> error ("initial_var_type_2: can't open "^source) in *)
(*   let finalise () =  *)
(*     close_in fd; *)
(*     Hashtbl.iter  *)
(*       (fun var size ->  *)
(*         if not (Hashtbl.mem var_type_table var)  *)
(*      then Hashtbl.add var_type_table var (size, size)) *)
(*       size_table; *)
(*     var_type_table in *)
(*   let parse_array s =  *)
(*     let rgx = Str.regexp "\\[\\([0-9]+\\)\\]" in *)
(*     let rec aux s = *)
(*      try let _ = Str.search_forward rgx s 0 in *)
(*          let i = Str.matched_group 1 s in *)
(*          let k = Str.group_end 1 in *)
(*          i::(aux (String.sub s k ((String.length s)-k)))  *)
(*      with Not_found -> [] in *)
(*     aux s in *)
(*   let rec parse () = *)
(*     let line = input_line fd in *)
(*     if (try ignore(Str.search_forward (Str.regexp "func_\\|main") line 0); true  *)
(*         with Not_found -> false) *)
(*     then finalise () *)
(*     else  *)
(*       (try *)
(*       if line.[0] = ' ' || line.[0] = '/' || line.[0] = '#' || line = "" *)
(*       then parse ()  *)
(*       else begin *)
(*         debug ("*** parsing "^line);  *)
(*         let ll = Str.split (Str.regexp " ") line in *)
(*         let rec skip_qualifier ll = *)
(*           match List.hd ll with  *)
(*             | "volatile" | "static" -> skip_qualifier (List.tl ll) *)
(*             | _ -> ll in *)
(*         let ll = skip_qualifier ll in *)
(*         let typ = *)
(*           match ll with *)
(*             | t1::v1::_ -> if v1.[0] = '*' then t1^" *" else t1 *)
(*             | _ -> raise Not_found *)
(*         in *)
(*         debug ("type: "^typ); *)
(*         let parse_var vs = *)
(*           let vs = *)
(*             if vs.[0] = '*' then String.sub vs 1 ((String.length vs) - 2) else vs in *)
(*           let vs = *)
(*             if vs.[(String.length vs)-1] = ';'  *)
(*             then String.sub vs 0 ((String.length vs) - 1) else vs in *)
(*           if String.contains vs ','  *)
(*           then error "initial_var_type_2: , separated variables not supported"; *)
(*           let vs =  *)
(*             if String.contains vs '['  *)
(*             then let l = String.index vs '[' in  *)
(*                  String.sub vs 0 l,  *)
(*                  Some (parse_array (String.sub vs l ((String.length vs)-l))) *)
(*             else vs, None *)
(*           in vs in *)
(*         let var,arr = parse_var (List.hd (List.tl ll)) in *)
(*         debug ("var: "^var); *)
(*         (match arr with None -> () *)
(*           | Some il -> debug ("array: "^(String.concat " " il))); *)
(*         (\* NOW PUT THINGS IN THE TABLE *\) *)
(*         let var_type_size =  *)
(*           (type_to_size typ) * (array_to_size arr) in *)
(*         debug ("var_type_size = "^(string_of_int var_type_size));  *)
(*         if var.[0] = 'g' || var.[0] = 'c' || var.[0] = 'a' then begin *)
(*              let var_size = *)
(*                try Hashtbl.find size_table var  *)
(*                with Not_found -> error  *)
(*               ("initial_var_type : " ^ var ^ " not found in original symbol table") *)
(*              in *)
(*              Hashtbl.add var_type_table var (var_type_size, var_size); *)
(*         end; *)
(*         parse () *)
(*       end *)
(*        with Not_found -> finalise ()) *)
(*   in parse () *)



(* called before calling initial_state, this creates a hashtable with
var_name -> (var_type_size*var_size) from the symbol_table of the object file *)
(*TODO : make it tail recursive*)
(* let initial_var_type executable size_table =  *)
(*   let var_type_table = Hashtbl.create 15 in *)
(*   let fd =  *)
(*     try Unix.open_process_in ("structure_layout.pl g_ "^executable ^ " 2> /dev/null")  *)
(*     with _ -> error ("structure_layout of "^executable^ "failed.") *)
(*   in *)
(*   let rec parse () =  *)
(*     try *)
(*       let line = input_line fd in *)
(*       let info_list = (Str.split (Str.regexp " ") line) in *)
(*       let var_name = (List.nth info_list 0) in *)
(*       if var_name.[0] = 'g' || var_name.[0] = 'c' || var_name.[0] = 'a' then begin *)
(*         let var_type_size = int_of_string (List.nth info_list 1) in  *)
(*         let var_size = *)
(*           try Hashtbl.find size_table var_name  *)
(*           with Not_found -> error ("initial_var_type : " ^ var_name ^ " not found in original symbol table") *)
(*         in *)
(*         Hashtbl.add var_type_table var_name (var_type_size, var_size); *)
(*       end; *)
(*       parse () *)
(*     with End_of_file ->  *)
(*       ignore (Unix.close_process_in fd); *)
(*       Hashtbl.iter (fun var size ->  *)
(*         if not (Hashtbl.mem var_type_table var ) then  *)
(*           Hashtbl.add var_type_table var (size, size) *)
(*           ) size_table; *)
(*       var_type_table *)
(*   in parse () *)


(* (\* called before calling initial_state, this creates a hashtable with *)
(* var_name & size from the symbol_table of the object file *\) *)

(* let initial_var_size executable arch =  *)
(*   let var_size_table = Hashtbl.create 15 in *)
(*   let fd =  *)
(*     try *)
(*       ( match arch with *)
(*      | X86 -> *)
(*        Unix.open_process_in ("readelf -s " ^ executable ^ " 2> /dev/null | grep 'g_\\|csmith\\|crc32\\|a_'") *)
(*      | ARM -> *)
(*        Unix.open_process_in ("arm-elf-readelf -s " ^ executable ^ " 2> /dev/null | grep 'g_\\|csmith\\|crc32\\|a_'"))  *)
(*     with _ ->  error ("initial_var_size, readelf of "^executable^"failed.") in *)
(*   let rec parse () =  *)
(*     try *)
(*       let line = input_line fd in *)
(*       let info_list = List.filter (fun x -> match x with "" -> false | _ -> true)  *)
(*           (Str.split (Str.regexp " ") line) in *)
(*       let var_name = (List.nth info_list 7) in *)
(*       if var_name.[0] = 'g' || var_name.[0] = 'c' || var_name.[0] = 'a' then begin *)
(*         let var_size = int_of_string (List.nth info_list 2) in  *)
(*         Hashtbl.add var_size_table var_name var_size *)
(*       end; *)
(*       parse () *)
(*     with End_of_file ->  *)
(*       ignore (Unix.close_process_in fd); *)
(*       var_size_table *)
(*   in parse () *)
