(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

open String
open Types
open Util

(* parsing of raw traces *)

type parse_line_res =
  | Line of raw_event
  | IReads of int list
  | LDList of (int*int list) list
  | Copts of string
  | End
  | ErrorTooBig

let rec parse_line fd : parse_line_res =
  try
    let line = input_line fd in
    if line.[0] = '=' 
    then parse_line fd 
    else if line.[0] = 'F' then 
      Line RFlush
    else if line.[0] = '*' then begin
      match line.[1] with
	| 'I' ->
	  let l = Str.split (Str.regexp ":") line in
	  if List.length l = 1 then IReads [] 
	  else
	    let idxs = Str.split (Str.regexp " ") (List.nth l 1) in
	    if List.length idxs > 100000 
	    then 
	      ErrorTooBig
	    else 
	      let idx = List.map int_of_string idxs in
	      IReads idx 
	|  'L' ->
	  let l = String.sub line 8 ((String.length line)-8) in
	  let ls = Str.split (Str.regexp ";;") l in
	  let lss = List.map 
	    (fun s -> Str.split (Str.regexp ":") s)
	    ls in
	  let ldlist = 
	    opt_map (fun (x,y) ->
	      (int_of_string (strip x),
	       List.map (fun z -> int_of_string (strip z)) (Str.split (Str.regexp " ") y)))
	      (List.map list_to_pair lss) in
	  LDList ldlist
	| 'C' -> 
	  Copts (String.sub line 7 ((String.length line)-7))
	| _ -> 	
	  warning ("cannot parse line: "^line); 
	  parse_line fd
    end else begin  
      (* standard trace entries *)
      let l = Str.split (Str.regexp ":") line in
      match (List.hd l).[0] with
      | 'I' | 'L' | 'S' | 'U' ->
        begin
          let address = try Int64.of_string ("0x"^(strip (List.nth l 1) )) with
            | Failure "int_of_string" -> error ("unable to convert \""^line^"\" to number at index 2")
            | Failure _ -> error ("error parsing "^line^" at index 1") in
	  match List.length l with
	    | 4 ->
	      let value = try Z.of_string ("0x"^(strip (List.nth l 2) )) with
		| Failure "int_of_string" -> error ("unable to convert \""^line^"\" to number at index 2")
		| Failure _ -> error ("error parsing "^line^" at index 3") in
	      let size = try int_of_string (strip (List.nth l 3) ) with
		| Failure "int_of_string" -> error ("unable to convert"^line^" to number at index 4")
		| Failure _ -> error ("error parsing "^line^" at index 4")  in
	      Line ( match (List.hd l).[0] with
		| 'L' -> RLoad (address,value,size)
		| 'S' -> RStore (address,value,size)
		| _ -> error "malformed trace - internal [1]" )
	    | 2 -> Line
	      ( match (List.hd l).[0] with
		| 'L' -> RLock address
		| 'U' -> RUnlock address
		| _ -> error "internal: malformed trace [3]" ) 
	    | _ -> error "malformed trace - internal [2]"
        end    
      | _ -> warning ("cannot parse line: "^line); parse_line fd 
    end
  with End_of_file -> close_in fd; End
    
let rec parse_file fd t irreads ldlist co =
  match parse_line fd with 
    | Line a -> parse_file fd (a::t) irreads ldlist co
    | IReads il -> parse_file fd t (il@irreads) ldlist co
    | LDList ll -> parse_file fd t irreads (ll@ldlist) co
    | Copts s -> parse_file fd t irreads ldlist s
    | End -> Some (List.rev t, irreads, ldlist, co)
    | ErrorTooBig -> close_in fd; None

(* TODO: the only exported functions *)
let parse_log (filename: string) : (raw_trace * int list * (int*int list) list * string) option =
  debug ("raw_parse_log "^filename);
  let fd = 
    try open_in filename 
    with Sys_error _ -> error ("can't open " ^ filename) in
  parse_file fd [] [] [] "" 
