(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

open Types
open Util
open Config

let cleanup = true

let replay_so = find_config "pin_so"

let same_file f1 f2 i =
  try
    let (fd1,fd2) = open_in f1, open_in f2 in
    let res =
      let rec aux ci =
	try
	  let l1 = input_line fd1 in
	  if l1.[0] = '*' then true
	  else begin
	    try
	      let l2 = input_line fd2 in
	      if ci = i then aux (ci+1)
	      else if l1.[0] = '=' then aux (ci+1)
	      else
		if String.compare l1 l2 = 0
		then aux (ci+1)
		else false
	    with End_of_file -> false (* can only happen if f2 is shorter than f1 -> replay problem *)
	  end
	with End_of_file -> error "same_file, eof fd1, never happen"
      in
      aux (-3)
    in
    (* Printf.printf "** same file %s %s %d: %b \n" f1 f2 i res; *)
    close_in fd1; close_in fd2;
    res
  with Sys_error _ -> error ("replay_same_file: file not found "^f1^" "^f2)

let replay objfile index =
  let replayed_trace = objfile^"_ir_"^(string_of_int index)^".log" in
  let rv = Util.exec_timeout 5 ("pin -t "^replay_so^" -mark -ir_index "^(string_of_int index)^" -o "^replayed_trace^" -- ./"^objfile^" > /dev/null 2>&1" ) in
  match rv with
    | Some _ ->
      let res = same_file (objfile^".raw_log") replayed_trace index in
      if cleanup then Sys.remove replayed_trace;
      res
    | None ->
      if cleanup then Sys.remove replayed_trace;
      false

let ir_analysis objfile trace =
  (* go over the trace *)
  (* - for each non redundant LOAD, invoke pin-replay on the load
   index, and compare the original trace with the new trace (only
   after that index *)
  (* - if traces are the same, mark the LOAD as IR *)
  let rec aux t i iri =
    match t with
      | [] ->
	(* Printf.printf "***** IR ANALYSIS *****\n %s \n\n" *)
	(*   (String.concat "," (List.map string_of_int iri)); *)
	iri
      | (Init _)::tl -> aux tl i iri
      | (Load _)::tl ->
	if (replay objfile i)
	then aux tl (i+1) (i::iri)
	else aux tl (i+1) iri
      | _::tl -> aux tl (i+1) iri
  in
  let res = aux trace 0 [] in
  if cleanup then Sys.remove (objfile^".raw_log");
  res
