(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Anirudh Kumar, IIT Kanpur & INRIA Paris-Rocquencourt                 *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

let prefix = ref ""
let syscall_prefix = ref ""
let llvm_svn = ref false
let verbose = ref false
let no_gcc = ref false
let cmmtest_only = ref false

let error s =
  print_endline ("install_cmmtest: "^s);
  exit 1

let print_log str = 
  if !verbose then print_endline str else ()

let syscall cmd =
  print_log (!syscall_prefix^": "^cmd);
  ignore (Sys.command ("echo '[["^cmd^"]]' >> "^ !prefix^"/install.log"));
  if Sys.command (cmd^" >> "^ !prefix^"/install.log 2>&1") != 0
  then error ("the last command failed,\nplease check: "
	      ^ !prefix^"/install.log")

let syscall_with_output cmd =
  let ic, oc = Unix.open_process cmd in
  let buf = Buffer.create 16 in
  (try
     while true do
       Buffer.add_channel buf ic 1
     done
   with End_of_file -> ());
  let _ = Unix.close_process (ic, oc) in
  (Buffer.contents buf)

let readfile f =
  let ic = open_in f in
  let n = in_channel_length ic in
  let s = String.create n in
  really_input ic s 0 n;
  close_in ic;
  s

let writefile str f = 
  let oc = open_out f in
  output_string oc str;
  close_out oc;
  print_log ("writing\n "^str^"\nto "^f)

let cd dir = 
  let _ = print_log("Changing to directory "^dir) in
  Sys.chdir dir

let notdir dir = 
  try not (Sys.is_directory dir)
  with Sys_error msg -> 
       let _ = print_log("Install: directory "^dir^"not found") in
       true
  
let notfile file =
  let _ = match not (Sys.file_exists file) with
        | true -> print_log(file^" was not found")
        | false -> print_log(file^" found") in
  not (Sys.file_exists file)
  
let install_framac prefix =
  print_log "INSTALL FRAMA C";
  syscall_prefix := "framac";
  syscall "wget -q http://frama-c.com/download/frama-c-Fluorine-20130501.tar.gz";
  syscall "tar -xzvf frama-c-Fluorine-20130501.tar.gz";
  Sys.remove "frama-c-Fluorine-20130501.tar.gz";
  syscall "rm -rf frama-c-bin/";
  syscall "mkdir -p frama-c-bin/";
  cd "frama-c-Fluorine-20130501/";
  syscall ("./configure --prefix="^prefix^"/frama-c-bin/");
  syscall "make";
  syscall "make install";
  cd "..";
  print_log "DONE INSTALL FRAMA C"


let download_gcc prefix =
  print_log "DOWNLOADING GCC";
  syscall_prefix := "gcc";
  syscall "rm -rf gcc-svn-src";
  syscall "svn checkout svn://gcc.gnu.org/svn/gcc/trunk gcc-svn-src";
  print_log "DONE DOWNLOADING GCC"


let install_gcc prefix = 
  print_log "INSTALLING GCC";
  syscall_prefix := "gcc";
  cd "gcc-svn-src";
  if notdir "mpc" || notdir "mpfr" || notdir "gmp" then
    syscall "./contrib/download_prerequisites";
  cd "..";
  syscall "rm -rf gcc-svn-obj/";
  syscall "rm -rf gcc-svn-bin/";
  syscall "mkdir -p gcc-svn-obj";
  syscall "mkdir -p gcc-svn-bin";
  cd "gcc-svn-obj";
  syscall ("../gcc-svn-src/configure --disable-bootstrap --enable-languages=c,c++ --prefix=" ^ prefix ^ "/gcc-svn-bin");
  syscall "make -j 4";
  syscall "make install";
  (* syscall ("export PATH=" ^ prefix ^ "/gcc-svn-bin/bin:$PATH"); *)
  (* syscall ("export LD_LIBRARY_PATH=" ^ prefix ^ "/gcc-svn-bin/lib64/:$LD_LIBRARY_PATH"); *)
  cd "..";
  print_log "DONE INSTALLING GCC"

let download_llvm_v3 prefix =
  print_log "DOWNLOADING LLVM";
  syscall_prefix := "llvm v3.3";
  syscall "wget -q http://llvm.org/releases/3.3/llvm-3.3.src.tar.gz";
  syscall "tar -xvzf llvm-3.3.src.tar.gz";
  syscall "rm -rf llvm-src";
  syscall "mv llvm-3.3.src llvm-src";
  syscall "rm -f llvm-3.3.src.tar.gz";
  cd "llvm-src/tools";
  syscall "wget -q http://llvm.org/releases/3.3/cfe-3.3.src.tar.gz";
  syscall "tar -xvzf cfe-3.3.src.tar.gz";
  syscall "rm -rf clang";
  syscall "mv cfe-3.3.src clang";
  syscall "rm -f cfe-3.3.src.tar.gz";
  cd "../..";
  cd "llvm-src/projects";
  syscall "wget -q http://llvm.org/releases/3.3/compiler-rt-3.3.src.tar.gz";
  syscall "tar -xvzf compiler-rt-3.3.src.tar.gz";
  syscall "rm -rf compiler-rt";
  syscall "mv compiler-rt-3.3.src compiler-rt";
  syscall "rm -f compiler-rt-3.3.src.tar.gz";
  cd "../..";
  print_log "DONE DOWNLOADING LLVM"

  
let download_llvm_svn prefix =
  print_log "DOWNLOADING LLVM SVN";
  syscall_prefix := "llvm svn";
  syscall "rm -rf llvm-svn-src";
  syscall "svn co http://llvm.org/svn/llvm-project/llvm/trunk llvm-svn-src";
  cd "llvm-svn-src/tools";
  syscall "svn co http://llvm.org/svn/llvm-project/cfe/trunk clang";
  cd "../..";
  cd "llvm-svn-src/tools/clang/tools";
  syscall "svn co http://llvm.org/svn/llvm-project/clang-tools-extra/trunk extra";
  cd "../../../..";
  cd "llvm-svn-src/projects";
  syscall "svn co http://llvm.org/svn/llvm-project/compiler-rt/trunk compiler-rt";
  cd "../..";
  print_log "DONE DOWNLOADING LLVM SVN"

let install_clang_llvm prefix =
  print_log "INSTALLING CLANG/LLVM";
  syscall_prefix := "llvm";
  syscall "rm -rf llvm-obj/";
  syscall "rm -rf llvm-bin/";
  syscall "mkdir -p llvm-obj";
  syscall "mkdir -p llvm-bin";
  cd "llvm-obj";
  syscall ("../llvm-src/configure --prefix=" ^ prefix ^ "/llvm-bin --enable-optimized");
  syscall "make -j 4";
  syscall "make install";
(*  syscall ("export PATH=" ^ prefix ^ "/llvm-bin/bin:$PATH"); *)
  cd "..";
  print_log "DONE INSTALLING CLANG/LLVM"


(* let install_ocaml prefix = *)
(*   syscall_prefix := "ocaml"; *)
(*   syscall "rm -rf ocaml-bin/"; *)
(*   syscall "mkdir -p ocaml-bin"; *)
(*   syscall "wget -q http://caml.inria.fr/pub/distrib/ocaml-4.00/ocaml-4.00.1.tar.gz"; *)
(*   syscall "tar -xzvf ocaml-4.00.1.tar.gz"; *)
(*   cd "ocaml-4.00.1"; *)
(*   syscall ("./configure -prefix " ^ prefix ^ "/ocaml-bin"); *)
(*   syscall "make world.opt"; *)
(*   syscall "make install"; *)
(*   syscall ("export PATH=" ^ prefix ^ "/ocaml-bin/bin:$PATH"); *)
(*   cd ".." *)

let install_csmith_atomics prefix =
  print_log "INSTALLING CSMITH ATOMICS";
  syscall_prefix := "csmith atomics";
  cd "csmith_with_atomics";
  syscall "rm -rf ../csmith-atomics-bin/";
  syscall "mkdir -p ../csmith-atomics-bin";
  syscall ("./configure --prefix="^prefix^"/csmith-atomics-bin/");
  syscall "make -j 4";
  syscall "make install";
(*  syscall ("export PATH=" ^prefix^ "/csmith-atomics-bin/bin:$PATH"); *)
  cd "..";
  print_log "DONE INSTALLING CSMITH ATOMICS"

  
let install_csmith_locks prefix =
  print_log "INSTALLING CSMITH LOCKS";
  syscall_prefix := "csmith locks";
  cd "csmith_with_locks";
  syscall "rm -rf ../csmith-locks-bin/";
  syscall "mkdir -p ../csmith-locks-bin";
  syscall ("./configure --prefix="^prefix^"/csmith-locks-bin/");
  syscall "make -j 4";
  syscall "make install";
(*  syscall ("export PATH=" ^prefix^ "/csmith-locks-bin/bin:$PATH"); *)
  cd "..";
  print_log "DONE INSTALLING CSMITH LOCKS"


let install_pin prefix =
  print_log "INSTALLING PIN";
  syscall_prefix := "pin";
  (* syscall "tar -xzvf pin-2.10-45467-gcc.3.4.6-ia32_intel64-linux.tar.gz"; *)
  syscall 
    ("wget -q http://software.intel.com/sites/landingpage/pintool/"
     ^ "downloads/pin-2.10-45467-gcc.3.4.6-ia32_intel64-linux.tar.gz");
  syscall "tar -xzvf pin-2.10-45467-gcc.3.4.6-ia32_intel64-linux.tar.gz";
  Sys.remove "pin-2.10-45467-gcc.3.4.6-ia32_intel64-linux.tar.gz";
  syscall "rm -rf pin";
  syscall "mv pin-2.10-45467-gcc.3.4.6-ia32_intel64-linux pin";
  cd "pin-interceptor2";
  writefile (Str.replace_first
	       (Str.regexp "^PIN_HOME \\?=.*") 
	       ("PIN_HOME ?= " ^ prefix ^ "/pin")
	       (readfile "Makefile"))
    "Makefile";
  syscall "make clean";
  syscall "make";
  cd "..";
  print_log "DONE INSTALLING PIN"


let install_creduce prefix =
  print_log "INSTALLING CREDUCE";
  syscall_prefix := "creduce";
  (*syscall "sudo apt-get install libfile-which-perl libregexp-common-perl
    indent astyle delta libexporter-lite-perl";*)
  cd "creduce-cmmtest";
  syscall "rm -rf ../creduce-bin/";
  syscall "mkdir -p ../creduce-bin";
  if !cmmtest_only 
  then syscall ("./configure --prefix="^prefix^"/creduce-bin")
  else syscall ("./configure --prefix="^prefix^"/creduce-bin --with-llvm="^prefix^"/llvm-bin/bin");
  syscall "make -j 4";
  syscall "make install";
  cd "..";
  print_log "DONE INSTALLING CREDUCE"


let install_cmmtest prefix =
  print_log "INSTALLING CMMTEST";
  syscall_prefix := "cmmtest scripts";
  cd "scripts";
  print_log "GENERATING THE CMMTEST CONFIG FILE";
  writefile ("csmith_include_dir="^prefix^"csmith-locks-bin/include/csmith-2.1.0\n"
    ^"pin_so = "^prefix^"/pin-interceptor2/obj-intel64/Interceptor.so\n"
    ^"gcc_x86_bin=gcc\n"
    ^"objdump_x86_bin=objdump\n"
    ^"readelf_x86_bin=readelf\n"
    ^"gcc_arm_bin=\n"
    ^"objdump_arm_bin=\n"
    ^"readelf_arm_bin=\n") "cmmtest.config";
  print_log "DONE GENERATING THE CMMTEST CONFIG FILE";
  syscall "make clean";
  (*writefile (Str.replace_first
	       (Str.regexp_string "| _ -> error \"unknown user\"") 
	       ("| _ -> \" -t " ^ prefix ^ "pin-interceptor/obj-intel64/Interceptor.so -mark \"")
	       (readfile "top.ml"))
    "top.ml";*)
  syscall "make";
  cd "..";
  print_log "DONE INSTALLING CMMTEST"


let install prefix =
  syscall_prefix := "install";
  if Sys.file_exists prefix then
    if not (Sys.is_directory prefix)
    then error "<prefix> must point to a directory, not to a file"
    else begin
      let rec user () =
	print_string 
	  ("install_cmmtest: the directory\n  "^prefix
	   ^"\nexists, do you want to proceed? (y)es/(n)o > ");
	match read_line () with
	  | "y" -> ()
	  | "n" -> exit 0
	  | _ -> user () in
      user () end;
  ignore (Sys.command ("mkdir -p "^prefix));
  if String.compare (Filename.basename (Sys.getcwd())) "scripts" != 0 then
    error "install_cmmtest should be run from inside the scripts/ directory";
  syscall ("cp -r ../* "^prefix);
  Sys.remove (prefix^"/install.sh");
  cd prefix;
  syscall ("rm install.log");
  (*  match Unix.fork() with
  | 0 -> 
  | pid ->  *)
  if not !cmmtest_only then begin
    (* LLVM *)
    if notfile (prefix^"/llvm-bin/bin/llvm-as") 
      || notfile (prefix^"/llvm-bin/bin/clang") then
      if !llvm_svn = true then download_llvm_svn prefix
      else download_llvm_v3 prefix;
    if notfile (prefix^"/llvm-bin/bin/llvm-as") 
      || notfile (prefix^"/llvm-bin/bin/clang") then
      install_clang_llvm prefix;
    (* Ocaml *)
    (* if notfile (prefix^"/ocaml-bin/bin/ocaml") then *)
    (* 	install_ocaml prefix; *)
    (* Frama-C *)
    if notfile (prefix^"/frama-c-bin/bin/frama-c") then 
      install_framac prefix;
    (* Csmith *)
    if notfile (prefix^"/csmith-atomics-bin/bin/csmith") then 
      install_csmith_atomics prefix;
    if notfile (prefix^"/csmith-locks-bin/bin/csmith") then
      install_csmith_locks prefix;
    (* GCC *)
    if not (!no_gcc) then begin
      if notfile (prefix^"/gcc-svn-bin/bin/gcc") then 
        download_gcc prefix;
      if notfile (prefix^"/gcc-svn-bin/bin/gcc") then 
        install_gcc prefix;
    end;
  end;
  (* Pin *)
  if notfile (prefix^"/pin-interceptor2/obj-intel64/cp-pin.exe") then
    install_pin prefix;
  (*     let _, _ = Unix.wait() in *)
  (* CMMTEST *)
  install_cmmtest prefix;
  (* CReduce *)
  if notfile (prefix^"/creduce-bin/bin/creduce") then
    install_creduce prefix;
(* FINAL OUTPUT *)
  print_log (syscall_with_output ("grep -A6 -B1 \"## WARNING: You cannot" 
    ^" run C-Reduce until you install missing dependencies! ##\" install.log"));

  print_endline ("******************************************************************\n");
  print_endline ("Use the following commands to set environment variables:\n");
  if not !cmmtest_only then begin
    if not (!no_gcc) then begin
      print_endline ("export PATH=" ^ prefix ^ "/gcc-svn-bin/bin:$PATH");
      print_endline ("export LD_LIBRARY_PATH=" ^ prefix ^ "/gcc-svn-bin/lib64:$LD_LIBRARY_PATH");
    end;
    print_endline ("export PATH=" ^ prefix ^ "/llvm-bin/bin:$PATH");
    print_endline ("export PATH=" ^prefix^ "/frama-c-bin/bin:$PATH");
    print_endline ("export PATH=" ^prefix^ "/csmith-locks-bin/bin:$PATH");
    print_endline ("export LD_LIBRARY_PATH=" ^ prefix ^ "/csmith-locks-bin/lib:$LD_LIBRARY_PATH");
  end;
  print_endline ("export PATH=" ^prefix^ "/creduce-bin/bin:$PATH");
  print_endline ("export PATH=" ^prefix^ "/pin:$PATH");
  print_endline ("export PATH=" ^prefix^ "/scripts:$PATH");
  print_endline ("\n******************************************************************\n");
  if not !cmmtest_only then begin
    print_endline ("Cmmtest invokes whatever version of CSmith is in the path.\n");
    print_endline ("if (and only if) you want to test atomics, then use the version below of CSmith instead:\n");
    print_endline ("export PATH=" ^prefix^ "/csmith-atomics-bin/bin:$PATH");
    print_endline ("export LD_LIBRARY_PATH=" ^ prefix ^ "/csmith-atomics-bin/lib:$LD_LIBRARY_PATH");
    print_endline ("\n******************************************************************\n");
  end


let _ =
  let options = 
    Arg.align 
      [("--prefix",
	Arg.String (fun s -> 
	  if not (Filename.is_relative s) then prefix := s 
	  else error "enter full path to directory"),
	"<directory> full path to the directory where you want to install cmmtest");
       ("--llvm_svn",
	Arg.Unit (fun () -> llvm_svn := true),
	" install clang and llvm from latest svn instead of version 3.3");
       ("--no-gcc",
	Arg.Unit (fun () -> no_gcc := true),
	" do not install the gcc svn");
       ("--cmmtest-only",
	Arg.Unit (fun () -> cmmtest_only := true),
	" install only the cmmtest tool");
       ("--verbose",
	Arg.Unit (fun () -> verbose := true),
	" display the complete install log")] in
  Arg.parse options
    (fun _ -> error "invalid option")
    "usage: install_cmmtest [options] <directory>";
  
  if !prefix = "" then
    error "specify --prefix <prefix>";

  if (!prefix).[(String.length !prefix) - 1] = '/' 
  then prefix := !prefix^"cmmtest"
  else prefix := !prefix^"/cmmtest";
 
  install !prefix
