(**************************************************************************)
(*                                 Cmmtest                                *)
(*                                                                        *)
(*   Robin Morisset, ENS & INRIA Paris-Rocquencourt                       *)
(*   Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt                  *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The Cmmtest tool is copyright 2012, 2013 Institut National de         *)
(*  Recherche en Informatique et en Automatique (INRIA).                  *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

open Str
open Types
open Util

(* *** trace marking algorithm *** *)

let opt_debug = ref false      (* INTERFACE *)

(* compares that the values x,y match for the minimum of size_x and size_y*) 
let compare_value x1 y1 = 
  match x1, y1 with
    | NonPointer x, NonPointer y -> x = y
    | Pointer x, Pointer y -> x = y
    | _ , _ -> false

let dump_table table st =
  print_endline "--------";
  Hashtbl.iter 
    (fun l (s,a,v,rel) ->
      let loc = string_of_location st l in
      if loc.[0] = 'g' || loc.[0] = 'a' then
        Printf.printf "%s: %d %b\n"
          loc s rel
    ) table;
  print_endline "--------"

let set_release table =
  (* question for OCaml expert, do we have to copy the table here? *)
  let table_copy = Hashtbl.copy table in
  Hashtbl.iter 
    (fun k (a,v,rel) -> 
      Hashtbl.replace table k (a,v,true))
    table_copy

let set_acquire table =
  (* if the release bit is set, then kill the dataflow, otherwise noop *)
  let table_copy = Hashtbl.copy table in
  Hashtbl.iter 
    (fun k (a,v,rel) -> 
      if rel then Hashtbl.remove table k)
    table_copy

let set_release_OW table =
  (* question for OCaml expert, do we have to copy the table here? *)
  let table_copy = Hashtbl.copy table in
  Hashtbl.iter 
    (fun k (ptr,ri,rel) -> 
      Hashtbl.replace table k (ptr,ri,true))
    table_copy

let set_acquire_OW table =
  (* if the release bit is set, then kill the dataflow, otherwise noop *)
  let table_copy = Hashtbl.copy table in
  Hashtbl.iter 
    (fun k (_,_,rel) -> 
      if rel then Hashtbl.remove table k)
    table_copy

(* for each location, var_addresses records the current value, 
 , if the last operation was a write or read, and if a release
   action has been encountered since the last access. *)
let var_addresses = (Hashtbl.create 15 : (loc_bo, (access * value * bool)) Hashtbl.t) 

let dump_var_addresses st =
  print_endline "--------";
  Hashtbl.iter (fun l (a,v,b) -> 
    let loc = string_of_loc_bo st l in
    if loc.[0] = 'g' || loc.[0] = 'a' then
    Printf.printf ": %s : %s - %s - %b\n" 
      loc (match a with R -> "R" | W -> "W" | I -> "I") (string_of_value v st) b)
    var_addresses;
  print_endline "--------"
      
let rec mark trace irreads (loaddeps:(int*int list)list) aft_sc st 
    : annot_event list = 
  match trace with
    | [] -> 
      []
    | ah :: t -> 
      ( (* dump_var_addresses st; *)
        match ah.evt with
      | Init (l,v,s) -> 
          let l = loc_bo_of_loc l in
          (* print_endline ("added "^(string_of_location l st)); *)
          Hashtbl.replace var_addresses l (I, v, false);
          ah :: (mark t irreads loaddeps false st)
      | Store (l,v,s) ->
          let l = loc_bo_of_loc l in
          (* print_endline ("\nmarking write "^(string_of_location st l)); *)
          if (List.length irreads > 0) && (ah.split_index == List.hd irreads) then begin
            Printf.printf "\nIR format**** split_index:%d  hd:%d\n" ah.split_index (List.hd irreads);
            assert(false);
          end;
          if Hashtbl.mem var_addresses l then
            let (op,oldV,rel) = Hashtbl.find var_addresses l in 
            match op with
              | R -> 
                if compare_value v oldV then begin 
                  ah.redundant := WAR;
                    (* TODO: unclear here *)
                  Hashtbl.replace var_addresses l (W, oldV, rel) 
               (* Hashtbl.replace var_addresses l (oldS, W, oldV, false) *)
                end else
                  Hashtbl.replace var_addresses l (W, v, false)
            | W | I -> 
              (* print_endline "in WAW"; *)
              (* write after write elimination should be correct when *)
              (* the latter write rewrites the same value and there is non r/a pair *)
              (* in between *)
              if compare_value v oldV
              then ah.redundant := WAW []; 
              Hashtbl.replace var_addresses l (W, v, false)
          else (
            (* print_endline "found nothing"; *)
            Hashtbl.replace var_addresses l (W, v, false) );
        ah :: (mark t irreads loaddeps false st)
      | Load (l,v,s)-> 
          let l = loc_bo_of_loc l in
        (* print_endline ("\nmarking load: "^(string_of_location st l)); *)
        (* first check for dependent IR *)
        (* warning: must apply -1 to get the right index during analysis *)
        let dll =
          List.concat (option_map (fun (x,y) -> if x = ah.split_index then Some y else None) loaddeps) in
        ( match dll with
          | [] -> ()
          | _ -> ah.redundant := IRR (List.map (fun x -> x-1) dll));
        (* the usual analysis *)
        if List.mem ah.split_index irreads 
        then ah.redundant := IRR [];
        if Hashtbl.mem var_addresses l then begin
          (* print_endline ("read found " ^ (string_of_location l st)); *)
          let (op,oldV,rel) = Hashtbl.find var_addresses l in 
          if op = R then
            ah.redundant := RAR
            (*no need to update the hash table entry *)
          else (* op was a W*)
            (ah.redundant := RAW;
               (*no need to compare values as our traces are sequential *)
               (* again, unclear here *)
             Hashtbl.replace var_addresses l (R, oldV, rel))
            (* Hashtbl.replace var_addresses l (R, oldV, false)) *)
        end else 
          Hashtbl.replace var_addresses l (R, v, false);
        (* for ARM, mark as UIL the u_ accesses *)
        if (string_of_loc_bo st l).[0] = 'u' 
        then ah.redundant := UIL;
        (* keep going *)
        ah :: (mark t irreads loaddeps false st)

      (* Clear any previous information at any flush or atomic instruction, except relaxed *)
      | AStore (a,l,v,s) -> 
        ( match a with
          | Relaxed -> ()
          | Release | Seq_cst -> set_release var_addresses );
        ah :: (mark t irreads loaddeps (is_seq_cst a) st)

      | ALoad (a,l,v,s) -> 
        ( match a with
          | Relaxed -> ()
          | Acquire | Seq_cst -> set_acquire var_addresses );
        (* TODO: why false here, instead of 'is_seq_cst a' *)
        ah :: mark t irreads loaddeps false st

      | Lock _ -> 
        set_acquire var_addresses;
        ah :: mark t irreads loaddeps false st

      | Unlock _ -> 
        set_release var_addresses;
        ah :: mark t irreads loaddeps false st

      | Flush -> 
        if not aft_sc then ah.redundant := RSF;
        ah :: (mark t irreads loaddeps false st)
      )

(* scan the trace, when it finds a W followed by redundant reads and
another store to the same location (and possibly other actions at
different locations) mark the W as OW with the list of indexed of
redundant reads *)
(* PROBLEM: how to deal with irrelevant reads here is unclear *)
(* ASSUME SIZE IS ALWAYS 1 *)
let mark_ow_writes st (trace:annot_event list) : unit = 
  let var_ow = (Hashtbl.create 15 : (int*int, (rr ref * int list * bool)) Hashtbl.t) in

  let dump_var_ow () =
    print_endline "--------";
    Hashtbl.iter 
      (fun (lb,lo) (_,_,rel) ->
        Printf.printf "%d,%d\n" lb lo
      ) var_ow;
    print_endline "--------" in

  (* to maintain the INIT count to correctly identify the indices as
     they are removed from array when passed to matchTraces *)
  let rec mark_ow (trace:annot_event list) index = 
    match trace with
      | [] -> ()
      | h :: t -> 
        (* Printf.printf "event: %s\n" (string_of_annot_event NoInit h st); *)
        match h.evt with
          | Init _ -> 
            mark_ow t index
          | Store (l,v,s) ->
            let l = loc_bo_of_loc l in
            if Hashtbl.mem var_ow l then begin
              let (ptr, redundant_indices, rel) = Hashtbl.find var_ow l in 
              if !ptr = NotRedundant || redundant_indices = [] then 
                ptr := OW redundant_indices;
              Hashtbl.replace var_ow l (h.redundant, [], rel)
            end else begin
              Hashtbl.replace var_ow l (h.redundant, [], false)
            end;
            mark_ow t (index+1)
          | Load (l,v,s) -> 
            let l = loc_bo_of_loc l in
            if Hashtbl.mem var_ow l then
              (match !(h.redundant) with
                | NotRedundant -> Hashtbl.remove var_ow l
                | _ -> let (rr, index_list, rel) = Hashtbl.find var_ow l in
                      Hashtbl.replace var_ow l (rr , index::index_list, rel));
            mark_ow t (index+1)
          (* flush is a noop for this dataflow *)  
          | Flush -> mark_ow t (index+1)
          (* Clear any previous information at any flush or atomic instruction *)
          | Lock _ -> 
            set_acquire_OW var_ow; mark_ow t (index+1)
          | Unlock _ -> 
            set_release_OW var_ow; mark_ow t (index+1)
          | ALoad (a,l,v,s) -> 
            ( match a with
              | Relaxed -> ()
              | Acquire | Seq_cst -> set_acquire_OW var_ow );
            mark_ow t (index+1)
          | AStore (a,l,v,s) -> 
            ( match a with
              | Relaxed -> ()
              | Release | Seq_cst -> set_release_OW var_ow );
            mark_ow t (index+1)
          (* | AStore _ | ALoad _ | Lock _ | Unlock _ -> *)
          (*   Hashtbl.clear var_ow; *)
          (*   mark_ow t (index+1) *)
  in
  mark_ow trace 0

let mark_WAW_chains_writes st (trace:annot_event list) : unit =
  let debug = false (* true *) in 
  let waw_chains = (Hashtbl.create 15 (* : ((int * int) (value * int list)) Hashtbl.t *) ) in
  let dump_waw_chains () =
    print_endline "WAW chains";
    let hdump = ref [] in 
    Hashtbl.iter (fun l (v,deps) -> hdump := (l,v,deps)::!hdump) waw_chains;
    let hdumpsorted = List.sort (fun ((_,o1),_,_) ((_,o2),_,_) -> o1-o2) !hdump in
    List.iter (fun ((l,o),v,deps) -> 
      let ls = (string_of_location st (l,o,[])) in
      if ls.[0] = 'c' then () else
        Printf.printf "%s: %s %s\n" ls (string_of_value v st) (String.concat "," (List.map string_of_int deps)))
      hdumpsorted;
    print_endline "*******" in
  let rec aux trace idx =
    match trace with
      | [] -> if debug then dump_waw_chains (); ()
      | ah :: t ->
        ( if debug then print_endline "---------";
          match ah.evt with
          | Init (l,v,s) ->
            let l = loc_bo_of_loc l in
            Hashtbl.replace waw_chains l (v,[]);
            aux t 0
          | Store (l,v,s) ->
            let l = loc_bo_of_loc l in
            if debug then dump_waw_chains ();
            if debug then print_endline (string_of_event st ah.evt);
            if Hashtbl.mem waw_chains l then
              let (ov,deps) = Hashtbl.find waw_chains l in
              if !(ah.redundant) = NotRedundant
              then begin
                if v = ov then begin
                  if debug then print_endline "WAW detected";
                  ah.redundant := WAW deps;
                  Hashtbl.replace waw_chains l (v,[]); (* WHAT DEPS HERE IS CRITICAL FOR THE HEURISTICS *)
                  aux t (idx+1)
                end else begin
                  Hashtbl.replace waw_chains l (v,[]);
                  aux t (idx+1)
                end end else begin
                  if debug then print_endline "already redundant";
                  Hashtbl.replace waw_chains l (ov,idx::deps);
                  aux t (idx+1)
                end
            else begin
                  Hashtbl.replace waw_chains l (v,[]);
                  aux t (idx+1)
            end
              
          | Load (l,v,s) ->
            let l = loc_bo_of_loc l in
            if debug then dump_waw_chains ();
            if debug then print_endline (string_of_event st ah.evt);
            if !(ah.redundant) = NotRedundant
            then begin
              Hashtbl.remove waw_chains l; 
              aux t (idx+1)
            end else begin
              let (ov,deps) = 
                try Hashtbl.find waw_chains l 
                with Not_found -> (v,[]) in
              Hashtbl.replace waw_chains l (ov,idx::deps);
              aux t (idx+1)
            end
          | ALoad _ | AStore _ | Flush | Lock _ | Unlock _ -> 
            aux t (idx+1) (* FIXME *)
        )
  in aux trace 0

let rec skip_init trace =
  match trace with
    | h::t when is_init h.evt -> skip_init t
    | _ -> trace

let merge_annot buf trace =
  let map_deps d = 
    ((List.nth (skip_init trace) d).split_index)-1 in
  let no_deps_write a =
    match a with
      | OW d | WAW d when not(d=[]) -> false
      | _ -> true in
  let merge_two a1 a2 =
    match a1,a2 with
      | NotRedundant, _ -> NotRedundant
      | _, NotRedundant -> NotRedundant
      | IRR _, _       -> a1 (* TODO, fix the dependencies *)
      | _, IRR _       -> a2 (* TODO, fix the dependencies *)
      | OW d1, OW d2   -> OW (d1@d2)
      | OW d, _        -> if no_deps_write a2 then a1 else CE
      | _, OW d        -> if no_deps_write a1 then a2 else CE
      | WAW d1, WAW d2 -> WAW (d1@d2)
      | WAW d, _       -> if no_deps_write a2 then a1 else CE
      | _, WAW d       -> if no_deps_write a1 then a2 else CE
      | _, _           -> if a1 = a2 then a1 else CE in
  if buf = [] then ()
  else
    let mr = List.fold_left merge_two (!(List.hd buf)) (List.map (!) (List.tl buf)) in
    let mrm = match mr with
      | OW d -> OW (Util.remove_duplicates (List.map map_deps d))
      | WAW d -> WAW (Util.remove_duplicates (List.map map_deps d))
      | _ -> mr  in 
    List.iter (fun x -> x := mrm) buf

let mark_merge trace st =
  let rec aux t ctr buf =
    match t with 
      | [] -> merge_annot buf trace
      | ah::tl when is_init ah.evt -> aux tl ctr []
      | ah::tl -> 
        if ah.split_index = ctr 
        then aux tl ctr (ah.redundant::buf)
        else begin
          merge_annot buf trace;
          aux tl ah.split_index [ah.redundant]
        end
  in aux trace 1 []

let  mark_ir trace irreads loaddeps : annot_event list =
  let rec aux trace last =
    match trace with
      | [] -> []
      | ah :: t ->
        ( match ah.evt with
          | Init (l,v,s) -> 
            ah :: (aux t None)
          | Load _ | ALoad _ -> 
            (* first dependent IR *)
            let dll = 
              List.concat (option_map (fun (x,y) -> if x = ah.split_index then Some y else None) loaddeps) in
            ( match dll with
              | [] -> ()
              | _ -> ah.redundant := IRR (List.map (fun x -> x-1) dll));
            (* and then real IR *)
            if List.mem ah.split_index irreads 
            then ah.redundant := IRR [];
            ( match ah.evt, last with
              | Load (l,v,s), Some (Load (l1,v1,s1)) when l = l1 && s = s1 && v = v1 -> 
                ah.redundant := RAR;
              | _,_ -> ());
            ah :: (aux t (Some ah.evt))
          | _ -> 
            ah :: (aux t None) )
  in aux trace None

(* *** trace comparison algorithm *** *)

let conflicting_events x y = 
  if x = Flush || y = Flush then false
  else loc_of_ev x = loc_of_ev y && (is_store x || is_store y)

(* checks if event x is swappable with event y.  Returns a pair of
   booleans a,b where a is true if x and y are swappable, and b is
   true if they are not swappable but x is eliminable *)
(* TODO: to be extended when we have locks / flushes *)
(* TODO: for now this function is commutative, but it won't be
   commutative anymore when we have locks *)
(* TODO : assert the OW redundant list too *)
(* NB: this function has been inlined in 'search' for simplicity for now. *)

let same_op_loc_val =
  fun x y ->
    match x, y with
    | Flush, Flush -> true
    | Init (l1,v1,_), Init (l2,v2,_) -> same_loc l1 l2 && v1 = v2
    | Load (l1,v1,s1), Load (l2,v2,s2) 
    | Store (l1,v1,s1), Store (l2,v2,s2) -> same_loc l1 l2 && v1 = v2
    | ALoad (a1, l1, v1, _), ALoad (a2, l2, v2, _) -> same_loc l1 l2 && v1 = v2 && a1 = a2
    | AStore (a1, l1, v1, _), AStore (a2, l2, v2, _) -> same_loc l1 l2 && v1 = v2 && a1 = a2
    | Lock l1, Lock l2 -> same_loc l1 l2
    | Unlock l1, Unlock l2 -> same_loc l1 l2
    | _, _ -> false

(* reordering *)

let is_store = function Store _ -> true | _ -> false
let is_load = function Load _ -> true | _ -> false 
let is_same_action a b = (is_store a && is_store b) || (is_load a && is_load b)

let search (r:annot_trace) (deleted:bool array) (a:event) (i:int) =
  (* 1- search for an index j > i such that r.(j) is the same event as a *)
  let rec find_j j =
    if j = r.len then [] else
      if is_atomic (r.evts.(j)) && not (is_relaxed (r.evts.(j))) then [] 
      else if not deleted.(j) && same_op_loc_val r.evts.(j) a then
      (*check if j was OW *)
      (*problem only with overwritten writes as OW are marked as by going 
      backward while rest (RAR, RAW, ...) are marked by looking the trace forward*)
      match r.redundants.(j) with
      | OW indices -> (*find the list of possible matches and try again if one fails*)
        let ow_list =
          let rec find_ow k ow =
            if k = r.len then ow
            else if same_op_loc_val r.evts.(k) a then
              match r.redundants.(k) with
                | NotRedundant -> (k::ow)
                | _ -> find_ow (k+1) (k::ow)
            else
              match r.evts.(k), a with
                | Store (l1,_,_), Store (l2,_,_) ->
                  if l1 = l2 then
                    (* XXX : check when returned to have the same value before trying to match
                       OW S x 2
                       OW S x 3 (different value but still redundant)
                       S x 2
                    *)
                    match r.redundants.(k) with
                      | NotRedundant -> ow
                      | _ -> find_ow (k+1) (k::ow)
                  else find_ow (k+1) ow
                | _ , _  -> find_ow (k+1) ow
          in
          find_ow j []
        in
        ow_list
      | _ -> [j]
      else find_j (j+1) in
  (* 2- check if forall i <= z < j, a.(z) is swappable with a.(j) *)
  let rec check_j j z to_be_deleted =
    if z = j then Some j, to_be_deleted
    else if deleted.(z) then check_j j (z+1) to_be_deleted
    else if not (conflicting_events r.evts.(z) r.evts.(j))
    then check_j j (z+1) to_be_deleted
    else match r.redundants.(z) with
      | NotRedundant -> None, to_be_deleted
      | _ -> check_j j (z + 1) (z :: to_be_deleted)
  (* all together *)
  in 
  let rec ow_check_j m_list =
    match m_list with
      | [] -> None, []
      | h :: t ->
        match check_j h i [] with
          | None, _ -> ow_check_j t
          | Some z, to_be_deleted -> Some (List.rev m_list), to_be_deleted
  in 
  (* do not reorder an atomic access *)
  if (is_atomic a) && not (is_relaxed a) then None, []
  else ow_check_j (find_j (i + 1))

(* the naive recursive version of the trace matching algorithm *)
let reorder_elim_check opt_analyse_write_only (x: annot_trace) (y: annot_trace) =
  let lenX, lenY = x.len, y.len in

  let rec try_elim_in_x deleted_x index_x deleted_y index_y =
    (* Printf.printf "elim_x index: %d %d\n" index_x index_y; *)
    match x.redundants.(index_x) with 
      | NotRedundant -> false 
      | OW to_delete ->
            List.iter (fun i -> deleted_x.(i) <- true) to_delete;
            let old_deleted = deleted_x.(index_x) in
            deleted_x.(index_x) <- true;
            let temp = re_check deleted_x (index_x + 1) deleted_y index_y in
            List.iter (fun i -> deleted_x.(i) <- false) to_delete;
            deleted_x.(index_x) <- old_deleted;
            temp
      | IRR to_delete -> (
        (* List.iter (fun x -> print_endline (string_of_int x)) to_delete; *)
            (* FIXME take into accout dependencies on OW *)
            if List.for_all (fun i -> 
              match x.redundants.(i) with
                | NotRedundant -> false
                | OW (h::_) -> false
                | _ -> true) to_delete 
            then begin 
              List.iter (fun i -> deleted_x.(i) <- true) to_delete;
              let old_deleted = deleted_x.(index_x) in
              deleted_x.(index_x) <- true;
              let temp = re_check deleted_x (index_x + 1) deleted_y index_y in
              List.iter (fun i -> deleted_x.(i) <- false) to_delete;
              deleted_x.(index_x) <- old_deleted;
              temp end
            else false )
      | WAW deps ->
          if List.for_all (fun i -> deleted_x.(i)) deps 
          then begin
            let old_deleted = deleted_x.(index_x) in
            deleted_x.(index_x) <- true;
            let temp = re_check deleted_x (index_x + 1) deleted_y index_y in
            deleted_x.(index_x) <- old_deleted;
            temp
          end else
            false
      | _ -> 
        let old_deleted = deleted_x.(index_x) in
        deleted_x.(index_x) <- true;
        let temp = re_check deleted_x (index_x + 1) deleted_y index_y in
        deleted_x.(index_x) <- old_deleted;
        temp

  and try_elim_ir_in_y deleted_x index_x deleted_y index_y =
    match y.redundants.(index_y) with
      | IRR [] | RAR | OW [] -> re_check deleted_x index_x deleted_y (index_y + 1)
      | IRR to_delete -> 
        if List.for_all (fun i -> 
          match y.redundants.(i) with
            | NotRedundant -> false
            | OW (h::_) -> false
            | _ -> true) to_delete 
        then begin 
          List.iter (fun i -> deleted_y.(i) <- true) to_delete;
          let temp = re_check deleted_x index_x deleted_y (index_y + 1) in
          List.iter (fun i -> deleted_y.(i) <- false) to_delete;
          temp end
        else false
      | NotRedundant -> false
      | OW d -> false
      | _ -> error "internal: an element of the optimised trace has been marked
        eliminable with something else than an IR, OW, or RAR."

  and try_elim_flush_in_y deleted_x index_x deleted_y index_y =
    match y.evts.(index_y) with
      | Flush -> re_check deleted_x index_x deleted_y (index_y + 1)
      | _ -> false 

  and re_check deleted_x index_x deleted_y index_y =
    Printf.printf "re_check index: %d:%d %d:%d\n" lenX index_x lenY index_y;

    if lenX > index_x && deleted_x.(index_x)
    then re_check deleted_x (index_x + 1) deleted_y index_y
    else if lenY > index_y && deleted_y.(index_y)
    then re_check deleted_x index_x deleted_y (index_y + 1)

    (* analyse_write_only option *)
    else if opt_analyse_write_only && lenX > index_x && is_na_read (x.evts.(index_x))
    then re_check deleted_x (index_x + 1) deleted_y index_y
    else  if opt_analyse_write_only && lenY > index_y && is_na_read (y.evts.(index_y))
    then re_check deleted_x index_x deleted_y (index_y + 1) 

    (* main matching *)
    else
      match (lenX > index_x), (lenY > index_y) with
        | true, true ->
          print_endline "true,true";
          if same_op_loc_val x.evts.(index_x) y.evts.(index_y) then (
            (* same ops, try to match *)
            print_endline "try match";
            re_check deleted_x (index_x+1) deleted_y (index_y+1) )
          else
            (* different ops, try to reorder *)
            ( print_endline "try reorder";
              match search x deleted_x y.evts.(index_y) index_x with
              | None, _               -> false  (* cannot reorder *)
              | Some a, to_be_deleted_x ->        (* try reorder *)
                print_endline "found";
                List.exists (fun z ->
                  same_op_loc_val x.evts.(z) y.evts.(index_y)
                  && (
                    deleted_x.(z) <- true;
                    List.iter (fun i -> if i < z then
                        deleted_x.(i) <- not deleted_x.(i))
                      to_be_deleted_x;
                    let temp = re_check deleted_x index_x deleted_y (index_y + 1) in
                    deleted_x.(z) <- false;
                    List.iter (fun i -> if i < z then
                        deleted_x.(i) <- not deleted_x.(i))
                      to_be_deleted_x;
                    temp
                  )
                ) a
            )

            || try_elim_in_x deleted_x index_x deleted_y index_y
            || try_elim_ir_in_y deleted_x index_x deleted_y index_y
            || try_elim_flush_in_y deleted_x index_x deleted_y index_y
              
        (* unmatched x remaining but y is over *)
        | true, false -> try_elim_in_x deleted_x index_x deleted_y index_y
            
        (* unmatched y remaining but x is over *)
        | false, true -> 
          try_elim_ir_in_y deleted_x index_x deleted_y index_y 
          || try_elim_flush_in_y deleted_x index_x deleted_y index_y

        | false, false -> true
  in
  fun ix iy -> re_check (Array.make lenX false) ix (Array.make lenY false) iy

let rec remove_init_annot t =
  match t with
  | [] -> []
  | hd::tl -> match hd.evt with
    | Init _ -> remove_init_annot tl
    | _ -> t

let mk_annot_trace t =
    let l_evts = List.map (fun x -> x.evt) t 
    and l_redundants = List.map (fun x -> !(x.redundant)) t
    in
    let a_evts = Array.of_list l_evts
    and a_redundants = Array.of_list l_redundants
    in
    { evts = a_evts
    ; redundants = a_redundants
    ; len = Array.length a_evts
    }

(* OLD *)
let check_reorder_elim trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt ld_opt co_opt st dump = 
  let a_trace_unopt = mark trace_unopt ir_unopt ld_unopt false st in
  mark_ow_writes st a_trace_unopt;
  mark_WAW_chains_writes st a_trace_unopt;
  let a_trace_opt = mark_ir trace_opt ir_opt ld_opt in
  if dump != Quiet then begin
    print_endline ("\n\n*** unoptimised trace: " ^ co_unopt ^"\n");
    dump_annot_trace dump (a_trace_unopt,st);
    print_endline ("\n*** optimised trace: " ^ co_opt ^"\n");
    dump_annot_trace dump (a_trace_opt,st);
    print_endline "\n***\n";
  end;
  let init_removed_unopt = remove_init_annot a_trace_unopt in
  let init_removed_opt = remove_init_annot a_trace_opt in
  let unopt_arrays = mk_annot_trace init_removed_unopt in
  let opt_arrays = mk_annot_trace init_removed_opt in
  let result = reorder_elim_check false unopt_arrays opt_arrays 0 0 in
  print_endline (string_of_bool result);
  result

(* NEW STUFF HERE *)

(* project an unannotated trace on byte accesses *)

let project_byte st trace =
  let split (lb,lo,la) v s =
    let rec split_aux i evs =
      if i = s then evs else begin
        (* loc *)
        let nlo = lo + i in
        (* val *)
        match v with
          | NonPointer v ->
            let nv = Int64.logand (Int64.shift_right v (8*i)) (Int64.of_int 0xFF) in
            split_aux (i+1) (((lb,nlo,la),NonPointer nv)::evs)
          | Pointer l -> split_aux (i+1) (((lb,nlo,la),Pointer l)::evs)
      end
    in split_aux 0 [] in
  let split_event ev index =
    match ev with
      | Init (l,v,s) -> 
        List.map (fun (ls,vs) -> (Init (ls,vs,1), index)) (split l v s)
      | Load (l,v,s) -> 
        List.map (fun (ls,vs) -> (Load (ls,vs,1), index)) (split l v s)
      | Store (l,v,s) -> 
        List.map (fun (ls,vs) -> (Store (ls,vs,1), index)) (split l v s)
      | _ -> [ev, index]
  in
  let rec project_aux trace proj index = 
    match trace with
      | ev :: tl -> 
        project_aux tl 
          ((split_event ev index)::proj) 
          (match ev with Init _ -> 1 | _ -> index+1)
      | [] -> 
        List.map 
          (fun (ev,i) -> 
            { evt = ev;
              redundant = ref NotRedundant;
              split_index = i })
          (List.rev (List.flatten proj))
  in project_aux trace [] 1

(* project an annotated trace on byte accesses *)

let annot_project_byte dwarf st (trace:annot_event list) =
  let split (lb,lo,la) v s =
    let rec split_aux i evs =
      if i = s then evs else begin
        (* loc *)
        let nlo = lo + i in
        (* remap the attributes *)
        let nla = la in
          (* let type_tag = Dwarfparse.compute_type_tag dwarf (the (Dwarfparse.find_var_type_node dwarf lb)) in *)
          (* Dwarfparse.resolve_address type_tag lb nlo) in *)
        (* val *)
        match v with
          | NonPointer v ->
            let nv = Int64.logand (Int64.shift_right v (8*i)) (Int64.of_int 0xFF) in
            split_aux (i+1) (((lb,nlo,nla),NonPointer nv)::evs)
          | Pointer l -> split_aux (i+1) (((lb,nlo,la),Pointer l)::evs)
      end
    in split_aux 0 [] in
  let split_annot_event (ev:annot_event) index =
    match ev.evt with
      | Init (l,v,s) ->
        List.map (fun (ls,vs) -> (Init (ls,vs,1), !(ev.redundant), index)) (split l v s)
      | Load (l,v,s) ->
        List.map (fun (ls,vs) -> (Load (ls,vs,1), !(ev.redundant), index)) (split l v s)
      | Store (l,v,s) ->
        List.map (fun (ls,vs) -> (Store (ls,vs,1), !(ev.redundant), index)) (split l v s)
      | _ -> [ev.evt, !(ev.redundant), index]
  in
  let rec project_aux (trace:annot_event list) proj index =
    match trace with
      | ev :: tl ->
        project_aux tl
          ((split_annot_event ev index)::proj)
          (match ev.evt with Init _ -> 1 | _ -> index+1)
      | [] ->
        List.map
          (fun (ev,r,i) ->
            { evt = ev;
              redundant = ref r;
              split_index = i })
          (List.rev (List.flatten proj))
  in project_aux trace [] 1

(* copy the annotations to the original, un-splitted trace *)

(* TODO: fix complexity *)

let copy_annot trace byte_trace =
  let find_annot idx =
    let ev = List.find (fun e -> e.split_index = idx && not (is_init e.evt)) byte_trace in
    !(ev.redundant) in
  let rec aux trace idx acc = 
    match trace with
      | [] -> List.rev acc
      | h::t when is_init h -> 
        let nh = { evt = h; redundant = ref NotRedundant; split_index = 0 } in
        aux t idx (nh::acc)
      | h::t -> 
        let nh = { evt = h; redundant = ref (find_annot idx); split_index = 0 } in
        aux t (idx+1) (nh::acc)
  in aux trace 1 []

(* project an annotated trace on the different global variables *)

let collect_variables trace st =
  print_endline "collect variables";
  let rec aux trace vars =
    match trace with
      | ev :: tl -> 
        ( match ev.evt with
          | Init (l,_,_) -> 
            if String.length (string_of_location st l) > 2 then
              if (string_of_location st l).[1] = 'r' then aux tl vars else aux tl (l::vars)
            else aux tl vars
          | _ -> vars )
      | _ -> vars in
  aux trace []
 
let project_var st trace var =
  let rec aux trace proj = 
    match trace with 
      | [] -> List.rev proj
      | ev::tl -> 
        if is_atomic ev.evt then aux tl (ev::proj)
        else let (lb,lo,la) = loc_of_ev ev.evt in
             if (lb, 0, []) = var && not (is_init ev.evt)  (* CHECK *)
             then aux tl (ev::proj)
             else aux tl proj in
  aux trace []

let project trace_unopt trace_opt st =
  let vars = collect_variables trace_unopt st in
  Printf.printf "\nmatching: %s\n" (String.concat " " (List.map (string_of_location st) vars));
  List.map (fun var -> 
    (var, project_var st trace_unopt var, project_var st trace_opt var))
    vars

(* match the spine *)

let same_atomic e1 e2 =
  match e1, e2 with
    | ALoad (a1, l1, v1, s1), ALoad (a2, l2, v2, s2)
    | AStore (a1, l1, v1, s1), AStore (a2, l2, v2, s2) -> 
      l1 = l2 && v1 = v2 && a1 = a2 && s1 = s2
    | Lock l1, Lock l2 | Unlock l1, Unlock l2 -> 
      l1 = l2
    | _,_ -> false

let rec match_spine trace_unopt trace_opt =
  match trace_unopt, trace_opt with
    | ue::ut, oe::ot -> 
      if not (is_atomic ue) then match_spine ut trace_opt
      else if not (is_atomic oe) then match_spine trace_unopt ot
      else if same_atomic ue oe then match_spine ut ot
      else false
    | ue::ut, [] -> 
      if not (is_atomic ue) then match_spine ut [] else false
    | [], oe::ot -> 
      if not (is_atomic oe) then match_spine [] ot else false
    | [], [] -> true

exception Success
exception SpineError
exception MarkError
exception MatchError (* of loc *)

let trace_copy trace =
  let rec aux trace buf =
    match trace with
      | [] -> List.rev buf
      | h::t ->
        let nev = { evt = h; redundant = ref NotRedundant; split_index = 0 } in
        aux t (nev::buf) in
  aux trace []

let rec compare_marking t1 t2 =
  match t1,t2 with
    | h1::tl1, h2::tl2 -> 
      let m =
        ( match !(h1.redundant), !(h2.redundant) with
          | OW d1, OW d2 -> (List.sort compare d1) = (List.sort compare d2)
          | a1, a2 -> a1 = a2) in
      if m 
      then compare_marking tl1 tl2
      else false
    | [], [] -> true
    | _,_ -> false

let both_empty t1 t2 =
  let is_empty t1 =
    let t = skip_init t1 in 
    if List.length t = 0 then true else false in
  is_empty t1 && is_empty t2

let new_match opt_analyse_write_only 
    trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt ld_opt co_opt st dwarf dump : bool =
  (* marking the unopt trace *)
  let byte_trace_unopt = project_byte st trace_unopt in
  let annot_byte_trace_unopt = mark byte_trace_unopt ir_unopt ld_unopt false st in
  mark_ow_writes st annot_byte_trace_unopt;
  mark_WAW_chains_writes st annot_byte_trace_unopt;
  (* print_endline "\n*** BEFORE MERGE CHECK ***\n"; *)
  (* dump_annot_trace All (annot_byte_trace_unopt,st); *)
  mark_merge annot_byte_trace_unopt st;
  (* print_endline "\n*** AFTER MERGE CHECK ***\n"; *)
  (* dump_annot_trace NoInit (annot_byte_trace_unopt,st); *)
  let annot_trace_unopt = copy_annot trace_unopt annot_byte_trace_unopt in
  (* marking the opt trace *)
  let byte_trace_opt = project_byte st trace_opt in
  let annot_byte_trace_opt = mark_ir byte_trace_opt ir_opt ld_opt in
  mark_ow_writes st annot_byte_trace_opt;
  let annot_trace_opt = copy_annot trace_opt annot_byte_trace_opt in
  (* dump the traces *)
  dump_annot_traces dump 
    (annot_trace_unopt) co_unopt 
    (copy_annot trace_opt annot_byte_trace_opt) co_opt st;
  (* and now attempt to match *)

  let byte_annot_trace_unopt = annot_project_byte dwarf st annot_trace_unopt in 
  let byte_annot_trace_opt = annot_project_byte dwarf st annot_trace_opt in 
  (* dump the byte-traces *)
  dump_annot_traces dump
    (byte_annot_trace_unopt) co_unopt
    (byte_annot_trace_opt) co_opt st;

  let unopt_arrays_new = mk_annot_trace (remove_init_annot byte_annot_trace_unopt) in
  let opt_arrays_new = mk_annot_trace (remove_init_annot byte_annot_trace_opt) in
  let result_new = 
    reorder_elim_check opt_analyse_write_only unopt_arrays_new opt_arrays_new 0 0 in
  result_new
(*   try *)
(*     (\* empty traces match *\) *)
(*     if both_empty annot_trace_unopt annot_trace_opt then raise Success; *)
(*     (\* for debug purposes, match with the old marking alogrithm *\) *)
(*     let trace_unopt_copy = trace_copy trace_unopt in *)
(*     let old_annot_trace_unopt = mark trace_unopt_copy ir_unopt ld_unopt false st in *)
(*     mark_ow_writes old_annot_trace_unopt; *)

(*     let identical_marking = compare_marking annot_trace_unopt old_annot_trace_unopt in *)

(*     let unopt_arrays_old = mk_annot_trace (remove_init_annot old_annot_trace_unopt) in *)
(*     let opt_arrays_old = mk_annot_trace (remove_init_annot annot_trace_opt) in  *)
(*     let result_old = reorder_elim_check unopt_arrays_old opt_arrays_old 0 0 in *)
(*     (\* print_endline "\n*** OLD MARKING UNOPT ***\n"; *\) *)
(*     (\* dump_annot_trace NoInit (old_annot_trace_unopt,st); *\) *)

(* (\*    Printf.printf "\n mark: %b, new: %b, old: %b\n" identical_marking result_new result_old; *\) *)
(* (\*     if result_new != result_old then raise MatchError; *\) *)
(*     if result_new then raise Success else raise MatchError; *)

(*     (\* if result then raise (MatchError); *\) *)
  


(*     (\* first compare the two marking algorithms *\) *)
(*     (\* let trace_unopt_copy = trace_copy trace_unopt in *\) *)
(*     (\* let old_annot_unopt = mark trace_unopt_copy ir_unopt ld_unopt false st in *\) *)
(*     (\* mark_ow_writes old_annot_unopt; *\) *)
(*     (\* if not (compare_marking annot_trace_unopt old_annot_unopt)  *\) *)
(*     (\* then begin *\) *)
(*     (\*   (\\* print_endline "\n*** NEW MARKING PER BYTE ***\n"; *\\) *\) *)
(*     (\*   (\\* dump_annot_trace NoInit (annot_byte_trace_unopt,st); *\\) *\) *)
(*     (\*   print_endline "\n*** NEW MARKING ***\n"; *\) *)
(*     (\*   dump_annot_trace All (annot_trace_unopt,st); *\) *)
(*     (\*   print_endline "\n*** OLD MARKING ***\n"; *\) *)
(*     (\*   dump_annot_trace All (old_annot_unopt,st); *\) *)
(*     (\*   raise MarkError *\) *)
(*     (\* end; *\) *)
(*     (\* check the spine *\) *)
(*     (\* if not (match_spine trace_unopt trace_opt) then raise SpineError; *\) *)
(*     (\* print_endline "spine matched"; *\) *)
(*     (\* (\\* project on variables *\\) *\) *)
(*     (\* let var_traces = project annot_byte_trace_unopt annot_byte_trace_opt st in *\) *)
(*     (\* Printf.printf "projected: %d" (List.length var_traces); *\) *)

(*     (\* (\\* match per variable *\\) *\) *)
(*     (\* List.iter (fun (v,ut,ot) -> *\) *)
(*     (\*   let unopt_arrays = mk_annot_trace ut in *\) *)
(*     (\*   let opt_arrays = mk_annot_trace ot in *\) *)
(*     (\*   let result = reorder_elim_check unopt_arrays opt_arrays 0 0 in *\) *)
(*     (\*   if not result then raise (MatchError v)) *\) *)
(*     (\*   var_traces; *\) *)
(*     (\* raise Success *\) *)
(*   (\* result *\) *)
(*   with *)
(*     | Success -> print_endline "true"; true *)
(*     | SpineError -> print_endline "false - spine match failure"; false *)
(*     | MatchError (\* v *\) -> print_endline ("false - match failure "(\* ^(string_of_location st v)*\) ); false *)
(*     | MarkError -> print_endline "\nfalse - mark mismatch"; false *)


(* *** Interface *** *)

(* real work done in new_match *)
(* TODO REMOVE *)
(* let match_traces unoptlog trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt co_opt st dump = *)
(*   (\* HACK: for now do not try to match NOOPT and STDOPT *\) *)
(*   if ( *)
(*     try match search_forward (regexp "NOOPT\\|STDOPT") unoptlog 0 with | _ -> false *)
(*     with Not_found -> true ) *)
(*   then  *)
(*     let tmp_pass = Filename.basename (Filename.chop_extension unoptlog) in *)
(*     let pass =  *)
(*       try String.sub tmp_pass 0 ((String.length tmp_pass)-6)  *)
(*       with _ -> "" in *)
(*     Printf.printf "%70s : " pass;  *)
(*     (\* check_reorder_elim trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt co_opt st dump *\) *)
(*     new_match trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt co_opt st dump *)
(*   else  *)
(*     (Printf.printf "%77s\n" "skipped"; true) *)
    
(* count the number of atomic/non_atomic accesses *)
      
let stats trace =
  let rec count t nal nas nl ns nf nlock nunlock=
    match t with
      | [] -> 
        { no_aload = nal; no_astore = nas; no_load = nl; no_store = ns;
          no_flush = nf; no_lock = nlock; no_unlock = nunlock },
          nal+nas+nl+ns+nf+nlock+nunlock
      | h::t ->
        ( match h with
          | Load _ -> count t nal nas (nl+1) ns nf nlock nunlock
          | Store _ -> count t nal nas nl (ns+1) nf nlock nunlock
          | ALoad _ -> count t (nal+1) nas nl ns nf nlock nunlock
          | AStore _ -> count t nal (nas+1) nl ns nf nlock nunlock
          | Init _ -> count t nal nas nl ns nf nlock nunlock
          | Flush -> count t nal nas nl ns (nf+1) nlock nunlock
          | Lock _ -> count t nal nas nl ns nf (nlock+1) nunlock
          | Unlock _ -> count t nal nas nl ns nf nlock (nunlock+1))
  in count trace 0 0 0 0 0 0 0
  
let simple_stats logfile trace =
  let name = Filename.basename (Filename.chop_extension logfile) in
  let rec count t nal nas nl ns nf nlock nunlock=
    match t with
      | [] -> 
        Printf.printf 
          "%s: \t %d \t %d \t %d \t %d \t %d \t %d \t %d\n" 
          (align name 30) nal nas nl ns nf nlock nunlock; 
        flush stdout;
        nal+nas+nl+ns+nf+nlock+nunlock
      | h::t ->
        ( match h with
          | Load _ -> count t nal nas (nl+1) ns nf nlock nunlock
          | Store _ -> count t nal nas nl (ns+1) nf nlock nunlock
          | ALoad _ -> count t (nal+1) nas nl ns nf nlock nunlock
          | AStore _ -> count t nal (nas+1) nl ns nf nlock nunlock
          | Init _ -> count t nal nas nl ns nf nlock nunlock
          | Flush -> count t nal nas nl ns (nf+1) nlock nunlock
          | Lock _ -> count t nal nas nl ns nf (nlock+1) nunlock
          | Unlock _ -> count t nal nas nl ns nf nlock (nunlock+1))
  in count trace 0 0 0 0 0 0 0

let print_stat name stat = 
  Printf.printf 
    "%s: \t %d \t %d \t %d \t %d \t %d \t %d \t %d\n" 
    (align name 30) 
    stat.no_aload stat.no_astore stat.no_load stat.no_store
    stat.no_flush stat.no_lock stat.no_unlock

let print_stats name_unopt stat_unopt name_opt stat_opt =
  Printf.printf "\n%s: \t aload \t astore  load \t store \t flush \t lock \t unlock\n" (align "" 30); 
  print_stat name_unopt stat_unopt;
  print_stat name_opt stat_opt;
  flush stdout
  
(* INTERFACE *)
let analyse opt_ir_replay opt_timeout dump opt_write_only opt_real_quiet (trace_unopt,trace_opt,st,dwarf) = 
  (* match Trace_parse.parse_log unoptlog with  *)
  (*   | Some (trace_unopt, ir_unopt, ld_unopt, st_unopt, co_unopt) -> *)
  (*     let  Some (trace_opt, ir_opt, _, st, co_opt) = Trace_parse.parse_log_st optlog st_unopt in *)
  (*     let stat_unopt, nev = stats trace_unopt in *)
  (*     let stat_opt, _ = stats trace_opt in *)
  (*     let unoptname = Filename.basename (Filename.chop_extension unoptlog) in *)
  (*     let optname = Filename.basename (Filename.chop_extension optlog) in *)
  (*     let name =        *)
  (*    try String.sub unoptname 0 ((String.length unoptname)-6)  *)
  (*    with _ -> "" in *)
  (*     if nev < 10000  *)
  (*     then begin *)
  (*       let ir_opt =  *)
  (*    if opt_ir_replay then  *)
  (*      let obj = Filename.chop_extension optlog in *)
  (*      List.map (fun x -> x+1) (Irreplay.ir_analysis obj trace_opt)  *)
  (*    else ir_opt in *)
  (*    let outcome = *)
  (*      let f () =  *)
  (*        new_match  opt_write_only  *)
  (*          trace_unopt ir_unopt ld_unopt co_unopt trace_opt ir_opt co_opt st dump in *)
  (*      match opt_timeout with *)
  (*        | None -> Some (f()) *)
  (*        | Some t -> try Some (timeout f (float t)) with Timeout -> None in *)
  (*    if opt_real_quiet  *)
  (*    then begin *)
  (*      match outcome with *)
  (*        | None -> print_char '.' *)
  (*        | Some b -> if b then print_char '+' else print_char '-' *)
  (*    end *)
  (*    else begin *)
  (*      let outcome_string = *)
  (*        match outcome with *)
  (*          | None -> "timeout" *)
  (*          | Some b -> string_of_bool b in *)
  (*      print_stats unoptname stat_unopt optname stat_opt; *)
  (*      Printf.printf "%70s : %s\n" name outcome_string; *)
  (*    end; *)
  (*    flush stdout; *)
  (*    outcome *)
  (*     end else begin *)
  (*    if opt_real_quiet then print_char '.' *)
  (*    else begin *)
  (*      print_stats unoptname stat_unopt optname stat_opt; *)
  (*      Printf.printf "%77s\n" "skipped";  *)
  (*    end; *)
  (*    flush stdout; *)
  (*    None *)
  (*     end *)
  (*   | None -> if opt_real_quiet then print_char '_' else Printf.printf "%77s\n" "skipped (during parsing)"; None *)

  let stat_unopt, nev = stats trace_unopt.events in
  let stat_opt, _ = stats trace_opt.events in

  let unoptname = trace_unopt.executable in (* Filename.basename (Filename.chop_extension unoptlog) in *)
  let optname = trace_opt.executable in (* Filename.basename (Filename.chop_extension optlog) in *)
  let name = 
    try String.sub unoptname 0 ((String.length unoptname)-6)
    with _ -> "" in
  if nev < 10000
  then begin
    let ir_unopt,ld_unopt,co_unopt = trace_unopt.ir_reads, trace_unopt.load_deps, trace_unopt.comp_opts in
    let ir_opt,ld_opt,co_opt = trace_opt.ir_reads, trace_opt.load_deps, trace_opt.comp_opts in
      (* if opt_ir_replay then *)
      (*        let obj = Filename.chop_extension optlog in *)
      (*        List.map (fun x -> x+1) (Irreplay.ir_analysis obj trace_opt) *)
      (* else ir_opt *)
    let outcome =
      let f () =
        new_match opt_write_only
          trace_unopt.events ir_unopt ld_unopt co_unopt trace_opt.events ir_opt ld_opt co_opt st dwarf dump in
      match opt_timeout with
        | None -> Some (f())
        | Some t -> try Some (timeout f (float t)) with Timeout -> None in
    if opt_real_quiet
    then begin
      match outcome with
        | None -> print_char '.'
        | Some b -> if b then print_char '+' else print_char '-'
    end
    else begin
      let outcome_string =
        match outcome with
          | None -> "timeout"
          | Some b -> string_of_bool b in
      print_stats unoptname stat_unopt optname stat_opt;
      Printf.printf "%70s : %s\n" name outcome_string;
    end;
    flush stdout;
    outcome
  end else begin
    if opt_real_quiet then print_char '.'
    else begin
      print_stats unoptname stat_unopt optname stat_opt;
      Printf.printf "%77s\n" "skipped";
    end;
    flush stdout;
    None
  end


(* This is used for a fast analysis of atomic accesses - can probably be abandoned in favour of match_spine *)
 
let count_only (unoptlog, optlog) =  (* INTERFACE *)
  ()
  (* match Trace_parse.parse_log unoptlog with *)
  (*   | Some (trace_unopt, ir_unopt, ldunopt, st_unopt, _) -> *)
  (*     let Some (trace_opt, ir_opt, ldopt, st, _) = Trace_parse.parse_log_st optlog st_unopt in  *)
  (*     let rec count_only_int t rsc racq rrlx ssc srel srlx = *)
  (*    match t with *)
  (*      | [] ->  *)
  (*        Printf.printf  *)
  (*          " \t %d \t %d \t %d \t %d \t %d \t %d \n"  *)
  (*          rsc racq rrlx ssc srel srlx; flush stdout; *)
  (*        [rsc;racq;rrlx;ssc;srel;srlx] *)
  (*      | h::t -> *)
  (*        ( match h with *)
  (*          | Load _ | Store _ | Init _ | Flush | Lock _ | Unlock _ ->  *)
  (*            count_only_int t rsc racq rrlx ssc srel srlx *)
  (*          | ALoad (Seq_cst,_,_,_) ->               *)
  (*            count_only_int t (rsc+1) racq rrlx ssc srel srlx  *)
  (*          | ALoad (Acquire,_,_,_) ->               *)
  (*            count_only_int t rsc (racq+1) rrlx ssc srel srlx  *)
  (*          | ALoad (Relaxed,_,_,_) ->               *)
  (*            count_only_int t rsc racq (rrlx+1) ssc srel srlx  *)
  (*          | AStore (Seq_cst,_,_,_) ->              *)
  (*            count_only_int t rsc racq rrlx (ssc+1) srel srlx  *)
  (*          | AStore (Release,_,_,_) ->              *)
  (*            count_only_int t rsc racq rrlx ssc (srel+1) srlx  *)
  (*          | AStore (Relaxed,_,_,_) ->              *)
  (*            count_only_int t rsc racq rrlx ssc srel (srlx+1) ) *)
  (*     in  *)
  (*     Printf.printf "\t rsc \t racq \t rrlx \t ssc \t srel \t srel \n"; *)
  (*     let unopt_num = count_only_int trace_unopt 0 0 0 0 0 0 in *)
  (*     let opt_num = count_only_int trace_opt 0 0 0 0 0 0 in *)
  (*     let tmp_pass = Filename.basename (Filename.chop_extension unoptlog) in *)
  (*     let pass =  *)
  (*    try String.sub tmp_pass 0 ((String.length tmp_pass)-6)  *)
  (*    with _ -> "" in *)
  (*     Printf.printf "%60s : " pass;  *)
  (*     if compare unopt_num opt_num = 0  *)
  (*     then (print_endline "[co] true"; true)  *)
  (*     else (print_endline "[co] false"; false) *)
  (*   | None -> print_endline "skipped (during parsing)"; true *)
  
  

(* stats for the paper - TODO: not robust *)

let count_elim annot_trace_unopt =
  (* count now *)
  let t_c, raw_c, rar_c, ow_c, war_c, waw_c = ref 0, ref 0, ref 0, ref 0, ref 0, ref 0 in
  List.iter (
    fun ae -> 
      t_c := !t_c+1;
      match !(ae.redundant) with
      | RAW    -> raw_c := !raw_c + 1
      | RAR    -> rar_c := !rar_c + 1
      | OW  [] -> ow_c  := !ow_c + 1
      | WAR    -> war_c := !war_c + 1
      | WAW [] -> waw_c := !waw_c + 1 
      | _ ->   t_c := !t_c-1 )
    annot_trace_unopt;
  !t_c, [(RAW, !raw_c); (RAR, !rar_c); (OW [], !ow_c); (WAR, !war_c); (WAW [], !waw_c)]


let record_opt_stats unoptlog uts =   (* INTERFACE *)
  (* match Trace_parse.parse_log unoptlog with *)
  (*   | Some (trace_unopt, ir_unopt, ld_unopt, st, _) -> *)
  (*     let trace_length = List.length trace_unopt in *)
  (*     let filename, source_length = *)
  (*    match uts with  *)
  (*      | None -> *)
  (*        let filename =  *)
  (*          (String.sub (Filename.chop_extension unoptlog) 0 ((String.length (Filename.chop_extension unoptlog))-6) )^".c" in *)
  (*        filename, (count_lines filename)  *)
  (*      | Some u -> "*  ", u in *)
  (*     (\* marking the trace *\)  *)
  (*     let byte_trace_unopt = project_byte st trace_unopt in *)
  (*     let annot_byte_trace_unopt = mark byte_trace_unopt ir_unopt ld_unopt false st in *)
  (*     mark_ow_writes annot_byte_trace_unopt; *)
  (*     mark_merge annot_byte_trace_unopt st; *)
  (*     let annot_trace_unopt = copy_annot trace_unopt annot_byte_trace_unopt in *)
  (*     let tc, elim_stat = count_elim annot_trace_unopt in *)
  (*     let irc = List.length (List.filter (fun i -> !((List.nth annot_trace_unopt i).redundant) = NotRedundant) ir_unopt) in *)
  (*     ( match uts with  *)
  (*    | None -> *)
  (*      Printf.printf "%s : [source: %6d] [trace: %6d] [elim: %6d]" filename source_length trace_length (tc+irc); *)
  (*    | Some _ -> *)
  (*      Printf.printf "%s : [opt_r : %5d" filename (trace_length*100/source_length); *)
  (*      print_string "%] "; *)
  (*         Printf.printf "[trace: %6d] [elim: %6d]"  trace_length (tc+irc) ); *)
  (*     Printf.printf " [ratio: %3d" (((tc+irc)*100)/trace_length); print_string "%]"; *)
  (*     List.iter (fun (r,n) -> Printf.printf " [%s: %4d]" (string_of_rr r) n) ((IRR [], irc)::elim_stat); *)

  (*     let entry = { *)
  (*    source_size = source_length; *)
  (*    trace_size = trace_length; *)
  (*    no_eliminable = tc+irc; *)
  (*    no_elim_details = (IRR [], irc)::elim_stat; *)
  (*     } in *)
  (*     Types.stats := entry::!Types.stats; *)
  (*     print_endline ""; *)
  (*     (true, trace_length) *)
  (*   | None -> print_endline "skipped (during parsing)"; *) 
  (true,1)

let average l = 
  let rec aux l s t =
    match l with
      | [] -> s / t
      | f::r -> aux r (f+s) (t+1)
  in aux l 0 0

let median l =
  let l = List.sort (compare) l in
  let s = List.length l in
  if s mod 2 = 0 then ((List.nth l ((s/2)-1)) + (List.nth l  (s/2))) /2 
  else List.nth l (s/2)

let std_dev l a =
  let a = float a in
  let rec aux l s t =
    match l with
      | [] -> sqrt (s /. (float (t-1)))
      | f::r -> aux r (((float f)-.a)*.((float f)-.a) +. s) (t+1)
 in aux l 0. 0

let max l =
  let rec aux l m =
    match l with
      | f::r -> if f > m then aux r f else aux r m
      | [] -> m
  in aux l 0

let av_std_dev l = 
  let a = average l in
  let d = median l in 
  let m = max l in
  let s = std_dev l a in
  Printf.sprintf "%4d (%4d) (%4d) (%4.2f)" a d m s

let stats_process () =
  print_endline "*****************";
  Printf.printf "[avg. source   : %s]\n" (av_std_dev (List.map (fun e -> e.source_size) !Types.stats));
  Printf.printf "[avg. trace    : %s]\n" (av_std_dev (List.map (fun e -> e.trace_size) !Types.stats));
  Printf.printf "[avg. elim     : %s]\n" (av_std_dev (List.map (fun e -> e.no_eliminable) !Types.stats));

  List.iter (fun r ->
    Printf.printf "[avg. %s    : %s]\n" (string_of_rr r) (av_std_dev (List.map (fun e -> (List.assoc r e.no_elim_details)) !Types.stats)))
    [IRR[];RAW;RAR;OW [];WAR;WAW []]


(*  List.iter (fun x -> Printf.printf "%d " x.trace_size) !Types.stats *)
