//                                Cmmtest                             

//  Robin Morisset, ENS & INRIA Paris-Rocquencourt
//  Pankaj Pawan, IIT Kanpur & INRIA Paris-Rocquencourt  
//  Pankaj Prateek, IIT Kanpur & INRIA Paris-Rocquencourt  
//  Francesco Zappa Nardelli, INRIA Paris-Rocquencourt

// The Cmmtest tool is copyright 2012, 2013 Institut National de
// Recherche en Informatique et en Automatique (INRIA).

// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions 
// are met: 
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
// 3. The names of the authors may not be used to endorse or promote
// products derived from this software without specific prior written
// permission.

// THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS 
// OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY 
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
// OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
// IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include "pin.H"
#include <iostream>
#include <fstream>
#include <string.h>
#include <cstdlib>
#include <vector>
#include <map>
#include <set>
#include <cassert>
#include  <unordered_map>
#include "ir_reads.cpp"
#include "control_dep.cpp"
#include <utility>
#include <string>
#include <algorithm>

//Pankaj : Memory in general refers to Global memory and not the stack
//accesses..remember for pin its all the same. For now I have added a mask 
//to differentiate between a stack access and a global access. Its important 
//for detecting stack writes/reads using the corresponding address stored
//in register, specially incase of local pointers

//enum memory_access {StackRead, StackWrite};
//typedef struct _tracingUnit{
////  bool regOrStack;
//  INT32 noOfMemoryOperands;
//  //ADDRINT addr; //reg also mapped as number 
//  UINT64 functionIndex;
//  BOOL stackR;
//  BOOL stackW;
//  BOOL stackRW;
//  BOOL memoryR;
//  BOOL memoryW;
//  BOOl memoryRW;
//  BOOl branchTaken;
//  ADDRINT address;
//  BOOL globalMemory;
//  //for indirect calls, probably our usercode won't have but still 
//  //ADDRINT stackAddress2;  
//  //ADDRINT memoryAddress1;
//  //Not required as per my knowledge, if we have a Memory Write and 
//  //a Memory Read in the same instruction, then they correnspond to 
//  //the same address, ofcourse ignoring the string instructions
//  //ADDRINT memoryAddress2; 
//  ADDRINT ip;
//} TracingUnit;
//

// Introducing some global variables for testing compare dependencies
extern vector<pair<INT64, pair<bool, set<int> > > > control_dependency_structure;

//create at start of analysis call of an instruction and then delete at 
//the end of instruction to prevent stale information being passed on!
//TracingUnit *info;
extern set<int>used_index;
extern unordered_map<unsigned long, vector<int> > loads2stores;
extern bool debug;
extern vector<int> irr_index;
set<int> global_load_index;
//extern map<ADDRINT, vector<int> >access_table;
extern INT64 indexCount ;//number of dynamically executed global memory accesses or loads ...? 

extern bool ignore_reads_in_jumps;

BOOL do_not_instrument = false;

typedef unsigned char UChar;
typedef unsigned short UShort;
typedef unsigned int UInt;
typedef unsigned long ULong;
typedef unsigned long long ULongLong;

map<ADDRINT, bool>globalVariables;

UINT64 functionCount = 0; //keeps track of number of functions calls currently active on stack

vector<string> functionChain; //keeps track of the function call chain


//XXX : access_table structure ?
//map<int, tracingUnit* >


/* ================================================================== */
// Global variables 
/* ================================================================== */

UINT64 bblCount = 0;        //number of dynamically executed basic blocks
UINT64 threadCount = 0;     //total number of threads, including main thread
FILE * trace;
UINT64 insCount = 0;   //added replay

//#define STACK_MASK 0x700000000000
#define RODATA_MASK 0x400000

bool stack_read = false;
bool stack_write = false;

string fileName;  //added replay
INT64 indexG = 0;  //indices start at 0  //added replay

/* ===================================================================== */
// Command line switches
/* ===================================================================== */
KNOB<string> KnobOutputFile(KNOB_MODE_WRITEONCE,  "pintool",
    "o", "", "specify file name for Interceptor output");

KNOB<INT>   KnobIrIndex(KNOB_MODE_WRITEONCE,  "pintool",
    "ir_index", "-1", "replaces the load at the index to read a different value"); //added replay

KNOB<BOOL>   KnobCountIRReads(KNOB_MODE_WRITEONCE,  "pintool",
    "count", "false", "count irrelevant reads in the application");

KNOB<BOOL>   KnobCompare(KNOB_MODE_WRITEONCE,  "pintool",
    "compare", "false", "Print the control dependency list");

KNOB<BOOL>   KnobDebug(KNOB_MODE_WRITEONCE,  "pintool",
    "debug", "0", "print debug info");

KNOB<BOOL>   KnobIrrelevantReads(KNOB_MODE_WRITEONCE,  "pintool",
    "mark", "false", "marks the irrelevant reads using liveness analysis");

KNOB<BOOL>   KnobUnsoundIR(KNOB_MODE_WRITEONCE,  "pintool",
    "unsound", "false", "marks the reads used in jumps as irrelevant reads");

KNOB<BOOL>   KnobTraceBbl(KNOB_MODE_WRITEONCE, "pintool",
    "bt", "false", "show stats for the trace and basic blocks");
/* ===================================================================== */
// Utilities
/* ===================================================================== */

/*!
 *  Print out help message.
 */
INT32 Usage()
{
    cerr << "This tool traces the memory accesses of the executable, in a format suitable " << endl <<
            "for the cpptest application." << endl << endl;

    cerr << KNOB_BASE::StringKnobSummary() << endl;

    return -1;
}

/* ===================================================================== */
// Analysis routines
/* ===================================================================== */

/*!
 * Increase counter of the executed basic blocks and instructions.
 * This function is called for every basic block when it is about to be executed.
 * @param[in]   numInstInBbl    number of instructions in the basic block
 * @note use atomic operations for multi-threaded applications
 */

//Print a flush
VOID RecordFlush(VOID *ip){
  indexCount++;
  fprintf(trace, "Flush\n");
}

VOID RecordPthreadLock(ADDRINT mutex, ADDRINT return_addr){
  if(do_not_instrument)return;
  indexCount++;
  fprintf(trace, "Lock : %lx\n", mutex);
}

VOID RecordPthreadUnlock(ADDRINT mutex, ADDRINT return_addr){
  if(do_not_instrument)return;
  indexCount++;
  fprintf(trace, "Unlock : %lx\n", mutex);
}

VOID DebugFallCall(VOID *ip) {
	// Do nothing for call instructions
  /*fprintf(trace, "=CALL: %lx\n", (UINT64) ip);
  fflush(trace);*/
}

VOID DebugFall(VOID *ip) {
  fprintf(trace, "=FALL: %lx\n", (UINT64) ip);
  fflush(trace);
}

UChar revertTo; //added replay

//added replay
VOID RollBack (VOID *ip, VOID *addr, ADDRINT size){
  //cout<<"In RollBack function "<<indexG<<"   "<<indexCount<<"    "<<KnobIrIndex.Value()<<endl;
  if(KnobIrIndex.Value() != -1) {
    //cout<<"RollBack IF CONDITION    "<<indexG<<"   "<<indexCount<<"    "<<KnobIrIndex.Value()<<endl;
    if(KnobIrIndex.Value() == indexCount-1){
      UChar* value = (UChar *)addr;
      PIN_SafeCopy(value, &revertTo, 1);
      //fprintf(trace, "decrement : %x\n", *value); 
    }
    indexG++;
  }
}

// Print a memory read record
VOID RecordMemRead (VOID * ip, VOID * addr, ADDRINT size, ADDRINT ea_address) {

  if(KnobIrIndex.Value() != -1) {
    if (KnobIrIndex.Value() == indexCount) {  //added replay

      UChar* value = (UChar *)addr;
      //fprintf(trace, "before increment : %x\n", *value); 
      revertTo = *value;
      //UChar foo = 47;
      //PIN_SafeCopy(value, &foo, 1);
      (*value)++; 
      //fprintf(trace, "after increment : %x\n", *value); 
      //cout << "increment" << endl;
    }
  }

  //removes false positives stack accesses which involve writing on
  //stack by loading stack addresses in registers
  UINT64 result = (UINT64)addr & STACK_MASK;
  if (result) return;
  result = ( (UINT64)addr >> 20 ) ^ 0x4;
  if (!result) return;
  indexCount++;
  global_load_index.insert(indexCount);

  //fprintf(trace, "%lx\n", ea_address);

  if (size == 1) {
    UChar* value = (UChar *)addr;
    fprintf(trace,"Load  : %lx : %x : %lu\n", (ULong)addr, *value, size);
  }
  else if (size == 2){
    UShort* value = (UShort *)addr;
    fprintf(trace,"Load  : %lx : %x : %lu\n", (ULong)addr, *value, size);
  }
  else if (size == 4){
    UInt* value = (UInt *)addr;
    fprintf(trace,"Load  : %lx : %x : %lu\n", (ULong)addr, *value, size);
  }
  else if (size == 8){
    ULongLong *value = (ULongLong *)addr;
    fprintf(trace,"Load  : %lx : %llx : %lu\n", (ULong)addr, *value, size);
  }
  else if (size == 16){
    int not_split_128=1;
    if (not_split_128) {
      // Here we print a 128 bit access in its full glory
      fprintf(trace,"Load : %lx : ",(ULong)addr);
      UChar* value = (UChar*)addr; 
      char val[129];
      memset(val,'\0',sizeof(val));
      for (int j=0; j<16; j++) {
	for (int i=7; i>=0; i--) {
	  val[j*8+7-i] = ((*(value+j) >> i) & 1);
	}
      }
      for(int j=63;j>=0;j-=8) {
	ULongLong tmp1 = (val[j-3])*8 + (val[j-2])*4 + (val[j-1])*2 + (val[j]);
	ULongLong tmp2 = (val[j-7])*8 + (val[j-6])*4 + (val[j-5])*2 + (val[j-4]);
	fprintf(trace,"%llx%llx", tmp2, tmp1);
      }
      //      fprintf(trace," ");
      for(int j=127;j>=64;j-=8) {
	ULongLong tmp1 = (val[j-3])*8 + (val[j-2])*4 + (val[j-1])*2 + (val[j]);
	ULongLong tmp2 = (val[j-7])*8 + (val[j-6])*4 + (val[j-5])*2 + (val[j-4]);
	fprintf(trace,"%llx%llx", tmp2, tmp1);
      }
      fprintf(trace," : %lu\n", size);
    } else {
      // Here we split a 128 bit access into two 64 bits accesses
      // Morally wrong, but since it does not require teaching 
      // cpptest about 128bits integers it is the way to go (until when
      // OCaml provides a Int128 library)
      //
      // Two indexes need to be added to the global_load_index set for 
      // each mmx 128 bit instruction which is split into two 64 bit events
      indexCount++;
      global_load_index.insert(indexCount);  
      ULongLong* value = (ULongLong *)addr;
      fprintf(trace,"Load : %lx : %llx : %lu\n",(ULong)addr,*value,size);
      fprintf(trace,"Load : %lx : %llx : %lu\n",(ULong)addr+8,*(value+1),size);
    }
  }
  // else if (size == 16) {
  //   UInt *value = (UInt *)addr;
  //   fprintf(trace,"=Load of 128 bits  : %lx : %x : %lu\n", (ULong)addr, *value, size);
  // }
  else {
    fprintf(trace,"UNKNOWN SIZE : %lu at address %lx\n",size,(ULong)addr);
    //cout << "UNKNOWN SIZE : " <<  size << " at " << ip << endl;
  }
  if (debug)
    fprintf(trace,"=Index : R : %lx address : %lx ip : %lx : size : %lu\n", 
            indexCount,(ULong) addr, (ULong)ip, size);
}

// Print a memory write record

// here 2 is probably enough, should read the pin documentation again.
ULong temp_write_buffer[5];

VOID RecordMemWriteAddress (VOID * ip, UINT32 memOp, VOID * addr, ADDRINT size) {
  if (debug) {
    fprintf(trace,"=WriteAddress: storing %lx in memOp %d\n", (ULong) addr, memOp);
    fflush (trace); }

  // UINT64 result = ((UINT64)addr & STACK_MASK);
  // if (result) return;
  // result = ((UINT64)addr >> 20) ^ 0x4;
  // if (!result) return;

  assert (memOp < 5);
  temp_write_buffer[memOp] = (ULong) addr;

  if (debug) {
    fprintf(trace,"=WriteAddress: stored %lx in memOp %d\n", temp_write_buffer[memOp], memOp);
    fflush (trace); }
}

VOID RecordMemWriteValue(VOID * ip, UINT32 memOp, VOID * addr, ADDRINT size) {
  assert (memOp < 5);

  if (debug) {
    fprintf(trace,"=WriteValue: passed addr %lx for memOp %d\n", (ULong) addr, memOp);
    fflush (trace); }
  
  ULong addr_retr = temp_write_buffer[memOp];

  if (debug) {
    fprintf(trace,"=WriteValue: retrieved addr %lx for memOp %d\n", addr_retr, memOp);
    fflush (trace); }
  UINT64 result = ((UINT64)addr_retr & STACK_MASK);
  if (result) return;
  result = ((UINT64)addr_retr >> 20) ^ 0x4;
  if (!result) return;
  indexCount++;

  if (size == 1) {
    UChar* value = (UChar *)addr_retr;
    fprintf(trace,"Store : %lx : %x : %lu\n",(ULong)addr_retr, *value, size);
  }
  else if (size == 2){
    UShort* value = (UShort *)addr_retr;
    fprintf(trace,"Store : %lx : %x : %lu\n",(ULong)addr_retr, *value, size);
  }
  else if (size == 4){
    UInt* value = (UInt *)addr_retr;
    fprintf(trace,"Store : %lx : %x : %lu\n",(ULong)addr_retr, *value, size);
  }
  else if (size == 8){
    ULongLong *value = (ULongLong *)addr_retr;
    fprintf(trace,"Store : %lx : %llx : %lu\n",(ULong)addr_retr, *value, size);
  }
  else if (size == 16){
    int not_split_128=1;
    if (not_split_128) {
      // Here we print a 128 bit access in its full glory
      // Morally correct, but cpptest does not know what to do with it
      fprintf(trace,"Store : %lx : ",(ULong)addr_retr);
      UChar* value = (UChar*)addr_retr;
      char val[129];
      memset(val,'\0',sizeof(val));
      for (int j=0; j<16; j++) {
	for (int i=7; i>=0; i--) {
	  val[j*8+7-i] = ((*(value+j) >> i) & 1);
	}
      }
      for(int j=63;j>=0;j-=8) {
	ULongLong tmp1 = (val[j-3])*8 + (val[j-2])*4 + (val[j-1])*2 + (val[j]);
	ULongLong tmp2 = (val[j-7])*8 + (val[j-6])*4 + (val[j-5])*2 + (val[j-4]);
	fprintf(trace,"%llx%llx", tmp2, tmp1);
      }
      // fprintf(trace," ");
      for(int j=127;j>=64;j-=8) {
	ULongLong tmp1 = (val[j-3])*8 + (val[j-2])*4 + (val[j-1])*2 + (val[j]);
	ULongLong tmp2 = (val[j-7])*8 + (val[j-6])*4 + (val[j-5])*2 + (val[j-4]);
	fprintf(trace,"%llx%llx", tmp2, tmp1);
      }
      fprintf(trace," : %lu\n", size);
    } else {
      // Here we split a 128 bit access into two 64 bits accesses
      // Morally wrong, but since it does not require teaching 
      // cpptest about 128bits integers it is the way to go (until when
      // OCaml provides a Int128 library)
      ULongLong* value = (ULongLong *)addr_retr;
      fprintf(trace,"Store : %lx : %llx : %lu\n",(ULong)addr_retr,*value,size);
      fprintf(trace,"Store : %lx : %llx : %lu\n",(ULong)addr_retr+8,*(value+1),size);
      indexCount++;
    }
  }
  else {
    fprintf(trace,"UNKNOWN SIZE : %lu at address %lx\n",size,(ULong)addr_retr);
    // cout << "UNKNOWN SIZE : "  << size << endl;
  }
  if(debug)fprintf(trace,"=Index : W : %lx address : %lx ip : %lx\n", indexCount,(ULong) addr_retr, (ULong)ip);
  if(KnobIrIndex.Value() != -1) {
    indexG++;  //added replay
  }
}

VOID test_branch(bool taken, ADDRINT addr ){
  cout << hex << addr << " : " ;
  if(taken)cout << "branch taken" << endl;
  else cout << "not taken" << endl;
}


static bool checkIfXCHG(ADDRINT instr_addr){
  UChar temp = *(UChar *)(instr_addr);
  return (temp == 0x87)?true:false;
}


static bool checkIfLocked(ADDRINT instr_addr){
  //x86 instruction can have up to 4 prefixes each 1 byte long, in any order.

  /*0xF0: LOCK prefix
  0xF2: REPNE/REPNZ prefix
  0xF3: REP or REPE/REPZ prefix
  Prefix group 2
  0x2E: CS segment override
  0x36: SS segment override
  0x3E: DS segment override
  0x26: ES segment override
  0x64: FS segment override
  0x65: GS segment override
  Prefix group 3
  0x66: Operand-size override prefix
  Prefix group 4
  0x67: Address-size override prefix
   */

  int i = 4;
  UChar temp;
  while(i){
    temp = *(UChar *)(instr_addr + 4 - i);
    if(temp == 0xf0){return true;}
    switch(temp){
      case 0xf2:
      case 0xf3:
      case 0x2e:
      case 0x36:
      case 0x3e:
      case 0x26:
      case 0x64:
      case 0x65:
      case 0x66:
      case 0x67:
        break;
      default:
        return false;
    }
    i--;
  }
  return false;
}

VOID RecordCall(ADDRINT esp, ADDRINT addr, string function){
  if(debug){
    cout << hex <<  " esp : " << esp << endl;
    cout <<  hex<< "called " << function <<" from " << hex << addr<<endl;
    cout << dec;
    //functionChain.push_back(function);
  }
  if (debug && function.find("safe_") == 0) {
    cout << indexCount << endl;
    cout << "safe_ setting" << endl;
    //ignore_reads_in_jumps = true;
  }

  //XXX: call will push the return address, hence should be mapped correclty to functionCount, need to decide when to increment it
  functionCount++;
}

VOID RecordReturn(ADDRINT esp, ADDRINT addr, string function) {
  if (debug) {
    cout << hex << addr << " esp : " << esp << endl;
    cout << "leaving " << function <<endl;
    cout << dec;
    //assert(functionChain.back() == function);
    //functionChain.pop_back();
  }
  if (function == "main") {
    do_not_instrument = true;
  }
  functionCount--;
  if (debug && function.find("safe_") == 0) {
    //ignore_reads_in_jumps = false;
    cout << indexCount << endl;
    cout << "unsetting " << endl;
    for (auto is = used_index.begin(); is != used_index.end(); is++)
      cout << *is << " ";
    cout << endl;
  }
}

VOID RecordInstruction(ADDRINT addr) {
  if (debug) {
    fprintf(trace,"=At: %lx: \n", addr);
    fflush(trace);
  }
}


// Is called for every routine and instruments reads and writes
VOID Routine(RTN rtn, VOID *v){
  RTN_Open(rtn);
  char* func_name = const_cast<char*>(RTN_Name(rtn).c_str());
  //cout << func_name << endl;
  if(0 == strcmp(func_name, "pthread_mutex_lock")){
    RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)RecordPthreadLock,
        //IARG_ADDRINT, RTN_Address(rtn),
        IARG_FUNCARG_ENTRYPOINT_VALUE, 0,
        IARG_RETURN_IP,
        IARG_END);
  }
  if(0 == strcmp(func_name, "pthread_mutex_unlock")){
    RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)RecordPthreadUnlock,
        IARG_FUNCARG_ENTRYPOINT_VALUE, 0,
        IARG_RETURN_IP,
        IARG_END);
  }
  RTN_Close(rtn);
}



VOID InstrumentCallReturn(INS ins) {
  //for now killing the data flow on a function call...not sure if its correct to do so.
  if (INS_Category(ins) == XED_CATEGORY_CALL) {
    //access_table.clear();
    INS_InsertCall(ins, IPOINT_BEFORE, 
		   (AFUNPTR)RecordCall, IARG_REG_VALUE, REG_STACK_PTR,
		   IARG_INST_PTR,
		   IARG_ADDRINT, 
		   &RTN_Name(RTN_FindByAddress(INS_DirectBranchOrCallTargetAddress(ins))),
		   IARG_END);
    //cout << hex << INS_DirectBranchOrCallTargetAddress (ins) << endl;
    return;
  }
  if(INS_IsRet(ins)){
    //access_table.clear();
    INS_InsertCall(ins, IPOINT_BEFORE, 
        (AFUNPTR)RecordReturn, IARG_REG_VALUE, REG_STACK_PTR,
        IARG_INST_PTR,
        IARG_ADDRINT, 
        & RTN_Name(INS_Rtn(ins)),
        IARG_END);
    //cout << hex << INS_DirectBranchOrCallTargetAddress (ins) << endl;
    return;
  }
}

VOID InstrumentFlush(INS ins) {
  if(INS_LockPrefix(ins) || checkIfMfence(INS_Address(ins)) 
     ||  checkIfLocked(INS_Address(ins)) || checkIfXCHG(INS_Address(ins))) {
    INS_InsertPredicatedCall(ins,IPOINT_BEFORE, (AFUNPTR)RecordFlush,
			     IARG_CALL_ORDER, CALL_ORDER_LAST, 
			     IARG_INST_PTR, IARG_END);
    //XXX : verify the order in which it should be logged w.r.t the memory events
    //for this instruction.
  }
}

VOID Trace(TRACE tr, VOID* v) {
  string func_name = RTN_Name(TRACE_Rtn(tr));
  cout << "func_name in trace: " << func_name << endl;
  if (func_name.find("main") != 0 
      && func_name.find("func_") != 0 
      && func_name.find("memset") != 0
      && func_name.find("memcpy") != 0) 
    return;
  cout << "Starting trace address : 0x" << hex << TRACE_Address(tr) << endl;
  for( BBL bbl = TRACE_BblHead(tr); BBL_Valid(bbl); bbl = BBL_Next(bbl) ){
    cout << "\t " << BBL_Address(bbl) << endl;
    for( INS ins = BBL_InsHead(bbl); INS_Valid(ins) ; ins = INS_Next(ins)){
      cout << "\t\t " << INS_Address(ins) ;
      if (INS_IsBranch(ins))
        cout << " : " << INS_DirectBranchOrCallTargetAddress(ins);
      cout <<  endl;
    }
  }
}

// Is called for every instruction and instruments reads and writes
VOID Instruction(INS ins, VOID *v) {
  // Instruments memory accesses using a predicated call, i.e.
  // the instrumentation is called iff the instruction will actually be executed.
  //
  // The IA-64 architecture has explicitly predicated instructions. 
  // On the IA-32 and Intel(R) 64 architectures conditional moves and REP 
  // prefixed instructions appear as predicated instructions in Pin.


  //----analyse in detail operands in the instructions
  //analyse_instruction(ins);
  //return;
  if (INS_IsNop(ins)) return;
    
  //Tracing only function specific code. 
  RTN routine = INS_Rtn (ins);
  bool analyse_this_function = false;
  if (RTN_Valid(routine)) {
    char *func_name =
      const_cast<char *>(PIN_UndecorateSymbolName(RTN_Name(routine).c_str(), UNDECORATION_NAME_ONLY).c_str());

    if (debug)
      fprintf(trace,"=Function: %s, ", func_name);
    int x = strlen(func_name);
    //cout << func_name << endl;
    if(x > 18) {
      func_name[18] = 0;
      if(0 == strcmp(func_name, "std::__atomic_base")){
        analyse_this_function = true;
      }
      //TODO : not sure if these functions should be allowed to pass through the irrelevant read analysis ?
    }
    
    func_name[6]=0;
    if (!strcmp(func_name, "memset") || !strcmp(func_name, "memcpy")){
      analyse_this_function = true;
    } else {
      func_name[5]=0;
      if (!strcmp(func_name, "func_") /*|| !strcmp(func_name, "safe_")*/){
	analyse_this_function = true;
      } else {
	func_name[4]=0;
	if (!strcmp(func_name, "main")) {
	  analyse_this_function = true;
	}
      }
    }
    if (debug) {
      fprintf(trace," %s\n", analyse_this_function?"true":"false");
      fflush(trace); }

    if (!analyse_this_function) 
      return;
  }
  
  if (debug) {
    INS_InsertCall(ins, IPOINT_BEFORE, (AFUNPTR)RecordInstruction,
		   IARG_INST_PTR,
		   IARG_END);  
  }

  InstrumentFlush(ins);
  InstrumentCallReturn(ins);

  //XXX:should be below the irrelevant reads analysis call?
  //should be performed for every instruction
  //
  //UINT32 written_regs   = INS_MaxNumWRegs(ins); 
  //UINT32 read_regs      = INS_MaxNumRRegs(ins); 
  //dump_access_table(ins);
  UINT32 total_operands = INS_OperandCount(ins);

  set<int> *read_set= new set<int>;
  set<int> *write_set= new set<int>;
  set<int> *read_write_set= new set<int>;
  set<int>::iterator it;
  bool memory_read = false;
  bool memory_read_write = false;
  bool memory_write = false;
  if (KnobIrrelevantReads && KnobIrIndex.Value() == -1) {
    //bool stack_read = INS_IsStackRead(ins);
    //bool stack_write = INS_IsStackWrite(ins);
    //XXX : for irrelevant read detection it should not be a problem if we handle
    //stack read/writes along with the normal memory access? Think more.

    for (UINT32 i = 0; i < total_operands; i++) {
      //ignoring the implicit (register) operands is not sufficient. 
      // for e.g mov x, %eax          - (1) 
      //         test %eax, %eax      - (2)
      //         cmove y, %eax        - (3)
      //(1) is not redundant because (2) modifies the eflags {implicit operand for (2)}
      //read by (3), hence implicit operands can't be ignored atleast not the flag 
      //register, not still sure about %rsp, even %rax is implicit operand for many 
      //instructions. 

      if (INS_OperandIsImplicit(ins,i)) {
        //FIXME
        //string implicit_reg = REG_StringShort(INS_OperandReg(ins, i));
        //if(implicit_reg != "eflags" && implicit_reg != "rflags")
        //XXX: for now letting rflags pass
        //continue;
      }

      if (INS_OperandIsReg(ins,i)) {
        //if the operand is a register
        //mov eax, eax are treated as different operands
        REG reg = REG_FullRegName(INS_OperandReg(ins, i));
	//cout<<REG_StringShort(REG_FullRegName(reg))<<endl;
        //if(reg == esp || reg == eflags )continue;

        if (INS_OperandReadAndWritten(ins,i)) {
          //reg should exist in the table as it was read
          //XXX ignore the %rsp
          //read_set.insert(REG_FullRegName(reg));
          //write_set.insert(REG_FullRegName(reg));
          (*read_write_set).insert(REG_FullRegName(reg));
        } 
        else if (INS_OperandRead(ins,i)) {
          (*read_set).insert(REG_FullRegName(reg));
        }
        else if (INS_OperandWritten(ins,i)) {
          (*write_set).insert(REG_FullRegName(reg));
        }
      }
      else if (INS_OperandIsMemory(ins,i)) {
        REG basereg  = INS_OperandMemoryBaseReg(ins, i);
        REG indexreg = INS_OperandMemoryIndexReg(ins, i);

        if (basereg  != 0)
	  (*read_set).insert(REG_FullRegName(basereg));
        if (indexreg != 0)
	  (*read_set).insert(REG_FullRegName(indexreg));

        //stack reads and writes are not included in this
        //though x86 has instructions with 2 stores, can ignore them for now
        if (INS_OperandRead(ins,i) && INS_OperandWritten(ins, i)) {
          memory_read_write = true;  
        }
        else if(INS_OperandRead(ins,i)) {
          memory_read = true;
        }
        else if (INS_OperandWritten(ins,i)) {
          memory_write = true;
        }
        else {
          cout << "operand is memory but is neither read or written ...quitting."<< endl;
          assert(0);
        }
      }
    }

    // If the operation is xor r,r we kill the dataflow.  For this we
    // copy the read_write_set into the write_set, and clear both the
    // read_set and the read_write_set.
    if (INS_Opcode(ins) == XED_ICLASS_XOR) {
      if (INS_OperandCount(ins) == 3)
	if (INS_OperandIsReg(ins,0) && INS_OperandIsReg(ins,1)) {
	  REG reg0 = REG_FullRegName(INS_OperandReg(ins, 0));
	  REG reg1 = REG_FullRegName(INS_OperandReg(ins, 1));
	  if (reg0 == reg1) {
	    (*read_set).clear();
	    for (it=(*read_write_set).begin(); it!=(*read_write_set).end(); ++it)
	      (*write_set).insert (*it);
	    (*read_write_set).clear();
	  }
	}
    }
  }

  bool prevent = true; 
	/*  pankajp - Such instructions are detected (and not traced) in the function RecordMemRead()
		// do not trace local stores and loads
		if (INS_IsStackRead(ins) || INS_IsStackWrite(ins)) {
		  prevent = false;
		}
	*/

  UINT32 memOperands = INS_MemoryOperandCount(ins);
  for (UINT32 memOp = 0; prevent &&  (memOp < memOperands) ; memOp++) {
    // Note that in some architectures a single memory operand can be 
    // both read and written (for instance incl (%eax) on IA-32)
    // In that case we instrument it once for read and once for write.
    // if (INS_MemoryOperandIsRead(ins, memOp) && INS_MemoryOperandIsWritten(ins, memOp)) {
    //   INS_InsertPredicatedCall(
    //       ins, IPOINT_BEFORE, (AFUNPTR)RecordMemRead,
    //       IARG_INST_PTR,
    //       IARG_MEMORYOP_EA, memOp,IARG_MEMORYREAD_SIZE,
    //       IARG_MEMORYREAD_EA,
    //       IARG_END);
    //   if (INS_HasFallThrough(ins)) {
    //     INS_InsertPredicatedCall(
    //         ins, IPOINT_AFTER, (AFUNPTR)RecordMemWrite,
    //         IARG_INST_PTR,
    //         IARG_MEMORYOP_EA, memOp,IARG_MEMORYWRITE_SIZE,
    //         IARG_END);
    //   } else {
    //     cout << "NO Fall Through, written value can't be logged...STOP" << endl;
    //     exit(1);
    //   }
    // }
    // else 
    if(INS_MemoryOperandIsRead(ins, memOp)) {
      INS_InsertPredicatedCall(
          ins, IPOINT_BEFORE, (AFUNPTR)RecordMemRead,
          IARG_INST_PTR,
          IARG_MEMORYOP_EA, memOp, 
          IARG_MEMORYREAD_SIZE,
          IARG_MEMORYREAD_EA,
          IARG_END);
      //cout<<"Value of indexCount before "<<indexCount<<endl;
      if(KnobIrIndex.Value() == indexCount && INS_HasFallThrough(ins)) {
	//cout<<"Inserting Roll Back Call "<<indexCount<<"    "<<KnobIrIndex.Value()<<endl;
	INS_InsertPredicatedCall(
	  ins, IPOINT_AFTER, (AFUNPTR)RollBack,
	  IARG_INST_PTR,
	  IARG_MEMORYOP_EA, memOp, IARG_MEMORYREAD_SIZE,
	  IARG_END);
	//cout<<"Inserted Roll Back Call "<<indexCount<<"    "<<KnobIrIndex.Value()<<endl;
      }
      //cout<<"Value of indexCount after "<<indexCount<<endl;
    };
    if (INS_MemoryOperandIsWritten(ins, memOp)) {
      if (INS_HasFallThrough(ins)) {
        INS_InsertPredicatedCall(
            ins, IPOINT_BEFORE, (AFUNPTR)RecordMemWriteAddress,
            IARG_INST_PTR,
            IARG_UINT32, memOp,
            IARG_MEMORYOP_EA, memOp, 
            IARG_MEMORYWRITE_SIZE,
            IARG_END);
        INS_InsertPredicatedCall(
            ins, IPOINT_AFTER, (AFUNPTR)RecordMemWriteValue,
            IARG_INST_PTR,
            IARG_UINT32, memOp,
            IARG_MEMORYOP_EA, memOp, 
            IARG_MEMORYWRITE_SIZE,
            IARG_END);        
      } else if(INS_IsCall(ins)) {
	// pankajp - Call instructions do not have fall through, 
        // there is a jump to a location which is not known 
        // before the processing of the call instruction 
	INS_InsertPredicatedCall(
	    ins, IPOINT_BEFORE, (AFUNPTR) DebugFallCall,
	    IARG_INST_PTR,
	    IARG_END);
      } else {
       	INS_InsertPredicatedCall(
            ins, IPOINT_BEFORE, (AFUNPTR) DebugFall,
	    IARG_INST_PTR,
	    IARG_END);
      }
    }
  };

  //INS_InsertCall(ins, IPOINT_BEFORE, (AFUNPTR)test_branch,
  //    IARG_BRANCH_TAKEN, IARG_INST_PTR , IARG_END);  
  //  if(KnobIrrelevantReads){
  //    INS_InsertCall(ins, IPOINT_BEFORE, (AFUNPTR)UpdateInstrInfo,
  //        IARG_INST_PTR,
  //        // XXX assuming only 1 unique memory address per instruction
  //        IARG_ADDRINT, (memory_read  ? IARG_MEMORYREAD_EA : 0),
  //        IARG_ADDRINT, (memory_write ? IARG_MEMORYWRITE_EA : 0),
  //        IARG_ADDRINT, (memory_read_write ? IARG_MEMORYWRITE_EA : 0),
  //        IARG_END);
  //  }
      
  if (KnobIrrelevantReads && KnobIrIndex.Value() == -1) {
    
//    cout << "Instrumentation details " << endl;
//    cout << "\t" << hex << INS_Address(ins) <<endl; 
//    cout << "\t" << " memR : " << memory_read << "memW : " << memory_write << "memRW : " << memory_read_write << endl;
//    cout << dec;
//    
    assert ((*read_set).size() < 6);
    IARG_TYPE mem_address = IARG_MEMORYREAD_EA; 
    if (INS_IsMemoryRead(ins) && INS_IsMemoryWrite(ins)) {
      mem_address = IARG_MEMORYREAD_EA;
    }
    else if (INS_IsMemoryRead(ins))
      mem_address = IARG_MEMORYREAD_EA;
    else if (INS_IsMemoryWrite(ins)) {
      mem_address = IARG_MEMORYWRITE_EA;
    } else {
      if (memory_read || memory_write || memory_read_write) {
        if (debug) cout << "Failed at  " << hex<< INS_Address(ins) << dec << endl;
        assert(0);
      }
      else
        mem_address = IARG_INST_PTR;
    }

    if(KnobCompare) {
      INS_InsertCall(ins, IPOINT_BEFORE, (AFUNPTR)control_dependency_analysis,
	  IARG_CALL_ORDER, CALL_ORDER_LAST, 
	  IARG_BOOL, memory_read,
	  IARG_BOOL, memory_write,
	  IARG_BOOL, memory_read_write,
	  mem_address,
	  IARG_PTR, (void *)read_set,
	  IARG_PTR, (void *)write_set,
	  IARG_PTR, (void *)read_write_set,
	  IARG_BOOL, INS_IsMemoryRead(ins),
	  IARG_BOOL, INS_IsMemoryWrite(ins),
	  IARG_INST_PTR,
	  IARG_END);  
    }
		   

    INS_InsertCall(ins, IPOINT_BEFORE, (AFUNPTR)irrelevant_reads_analysis,
        IARG_CALL_ORDER, CALL_ORDER_LAST,
        IARG_BOOL, memory_read,
        IARG_BOOL, memory_write,
        IARG_BOOL, memory_read_write,
        mem_address,
        IARG_PTR, (void *)read_set,
        IARG_PTR, (void *)write_set,
        IARG_PTR, (void *)read_write_set,
        IARG_BRANCH_TAKEN, 
        // IARG_BOOL, INS_IsStackRead(ins),
        // IARG_BOOL, INS_IsStackWrite(ins),
	IARG_BOOL, INS_IsMemoryRead(ins),
        IARG_BOOL, INS_IsMemoryWrite(ins),
        IARG_INST_PTR,
        IARG_BOOL, INS_IsBranchOrCall(ins),
                   IARG_BOOL, (INS_RepPrefix(ins) || INS_IsStringop(ins)),
        IARG_END);  
  }
}

/* ===================================================================== */
   
VOID Image(IMG img, VOID *v)
{
    RTN rtn = RTN_FindByName(img, "pthread_mutex_lock");

    if (RTN_Valid(rtn)) {
      RTN_Open(rtn);
      RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)RecordPthreadLock,
          //IARG_ADDRINT, RTN_Address(rtn),
          IARG_FUNCARG_ENTRYPOINT_VALUE, 0,
          IARG_RETURN_IP,
          IARG_END);
      RTN_Close(rtn);
    }
    rtn = RTN_FindByName(img, "pthread_mutex_unlock");

    if (RTN_Valid(rtn)) {
      RTN_Open(rtn);
      RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)RecordPthreadUnlock,
          IARG_FUNCARG_ENTRYPOINT_VALUE, 0,
          IARG_RETURN_IP,
          IARG_END);
      RTN_Close(rtn);
    }
}

/* ===================================================================== */

/*!
 * Increase counter of threads in the application.
 * This function is called for every thread created by the application when it is
 * about to start running (including the root thread).
 * @param[in]   threadIndex     ID assigned by PIN to the new thread
 * @param[in]   ctxt            initial register state for the new thread
 * @param[in]   flags           thread creation flags (OS specific)
 * @param[in]   v               value specified by the tool in the 
 *                              PIN_AddThreadStartFunction function call
 */
VOID ThreadStart(THREADID threadIndex, CONTEXT *ctxt, INT32 flags, VOID *v) {
  threadCount++;
}

VOID dump_load2store() {
  fprintf(trace, "*LDList: "); 
  //FIXME : replace = with * to parse...currently treated as a comment
  for (auto it = loads2stores.begin(); it != loads2stores.end(); ++it) {
    fprintf(trace, "%ld : ", it->first) ; 
    for (auto is = (it->second).begin(); is != (it->second).end(); ++is) {
      fprintf(trace, "%d ", *is) ;
    }
    fprintf(trace, ";;");
  }  
  fprintf(trace, "\n");
}

/*!
 * Print out analysis results.
 * This function is called when the application exits.
 * @param[in]   code            exit code of the application
 * @param[in]   v               value specified by the tool in the 
 *                              PIN_AddFiniFunction function call
 */
VOID Fini(INT32 code, VOID *v) {
  if(KnobIrrelevantReads && KnobIrIndex.Value() == -1){
    vector<int>::iterator iv;
    set<int>::iterator is;
    set<int>::iterator it;
    //map<int,int>::iterator im;
    map<ADDRINT, vector<int> >::iterator im;
    for (is = used_index.begin() ; debug && is != used_index.end(); is++)
      cout << *is << " is used" << endl;
    for (is = global_load_index.begin(); is != global_load_index.end(); is++) {
      it = used_index.find(*is); 
      if (it == used_index.end()) {
        irr_index.push_back(*is);
      }
    }

    fprintf(trace, "*IReads:");
    if (KnobCountIRReads) {
      cout << irr_index.size() << endl;
    }
    //fprintf("%d", irr_index.size());
    for (unsigned int i = 0 ; i < irr_index.size(); i++) {
      fprintf(trace, "%d ", irr_index[i]);
      if (debug) cout << irr_index[i] <<"," ; 
    }

    fprintf(trace, "\n");

    if (debug) cout << endl;
    dump_load2store();
   
    if(KnobCompare) {
      print_control_dependencies();
    }
  }

  fclose(trace);
}

/*!
 * The main procedure of the tool.
 * This function is called when the application image is loaded but not yet started.
 * @param[in]   argc            total number of elements in the argv array
 * @param[in]   argv            array of command line arguments, 
 *                              including pin -t <toolname> -- ...
 */
int main(int argc, char *argv[]) {
    // Initialize PIN library. Print help message if -h(elp) is specified
    // in the command line or the command line is invalid 
    if( PIN_Init(argc,argv) )
    {
        return Usage();
    }

    //take in as input addresses of all global variables
    
    string executable = argv[argc - 1]; 
    //parse the readelf executable to generate vector<ADDRINT>globalVariables 
    //for now using STACK_MASK
    PIN_InitSymbols();

    if(KnobTraceBbl){
      TRACE_AddInstrumentFunction(Trace, 0); 
    }
    else{
      // Register function to be called to instrument traces
      INS_AddInstrumentFunction(Instruction,0);
      //RTN_AddInstrumentFunction(Routine, 0);
      IMG_AddInstrumentFunction(Image, 0);
    }


    // Register function to be called for every thread before it starts running
    PIN_AddThreadStartFunction(ThreadStart, 0);

    // Register function to be called when the application exits
    PIN_AddFiniFunction(Fini, 0);

    if (!KnobOutputFile.Value().empty()) 
      trace = fopen(KnobOutputFile.Value().c_str(), "w");
    else{ 
      if(debug)cout << "No log file specified, default : interceptor.out" << endl;
      trace = fopen("interceptor.out","w");
    }

    if(KnobDebug){
      debug = true;
    }
    
    if(KnobUnsoundIR){
      ignore_reads_in_jumps = true;
    }

    if(KnobIrIndex.Value() != -1) {
        fprintf(trace,"=================================================\n");
	fprintf(trace,"=This application is instrumented by REPLAY     =\n");
	fprintf(trace,"=================================================\n");
    } else {
      fprintf(trace,"=================================================\n");
      fprintf(trace,"=This application is instrumented by Interceptor=\n");
      fprintf(trace,"=================================================\n");
    }

    //Sets the disassembly syntax to XED detailed format which lists all 
    //resources read and written.
    //PIN_SetSyntaxXED();

    // Start the program, never returns
    ///will not compile
    PIN_StartProgram();

    return 0;
}

/* ===================================================================== */
/* eof */
/* ===================================================================== */
