(**************************************************************************)
(*                            arm-run & arm-tracer                        *)
(*                                                                        *)
(*   Pankaj More, IIT Kanpur & INRIA Paris-Rocquencourt                   *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The arm-run and arm-tracer tools are copyright 2012, 2013 Institut    *)
(*  National de Recherche en Informatique et en Automatique (INRIA).      *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(*  *******************************************************************   *)
(*                                                                        *)
(*  *******************************************************************   *)
(*                                                                        *)
(*                               arm-run                                  *)
(*                                                                        *)
(*  The arm-run tool is copyright 2010 - 2013 Anthony Fox, Magnus Myreen  *)
(*  and Mike Gordon, Computer Laboratory, University of Cambridge.        *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

val () = List.app PolyML.use
   ["combinML.sig", "combinML.sml",
    "pairML.sig", "pairML.sml",
    "sumML.sig", "sumML.sml",
    "numML.sig", "numML.sml",
    "listML.sig", "listML.sml",
    "setML.sig", "setML.sml",
    "optionML.sig", "optionML.sml",
    "rich_listML.sig", "rich_listML.sml",
    "basicSizeML.sig", "basicSizeML.sml",
    "stringML.sig", "stringML.sml",
    "bitML.sig", "bitML.sml",
    "fcpML.sig", "fcpML.sml",
    "wordsML.sig", "wordsML.sml",
    "intML.sig", "intML.sml",
    "sortingML.sig", "sortingML.sml",
    "patriciaML.sig", "patriciaML.sml",
    "armML.sig", "armML.sml"];

(* ------------------------------------------------------------------------ *)

local

infix |>

fun x |> f = f x

fun set_diff a b = List.filter (fn c => not (List.exists (fn d => c = d) b)) a

fun set_union a b = (set_diff a b) @ b

local
   fun merge cmp =
      let
         fun mrg ([], a) = a
           | mrg (a, []) = a
           | mrg (A as (a::t1), B as (b::t2)) =
               if cmp (a, b) = General.LESS
                  then a::mrg(t1, B)
               else b::mrg(A, t2)
      in
         mrg
      end
    fun loop (x::y::zs, xs, ys) = loop (zs, x::xs, y::ys)
      | loop (x::[], xs, ys) = (x::xs, ys)
      | loop ([], xs, ys) = (xs, ys)
   fun split ns = loop (List.rev ns, [], [])
in
   fun msort cmp xs =
      let
         val merge' = merge cmp
         fun ms [] = []
           | ms [x] = [x]
           | ms xs =
               let
                  val (left, right) = split xs
               in
                  merge' (ms left, ms right)
               end
      in
         ms xs
      end
end

(* ------------------------------------------------------------------------ *)

type arm_mem = armML.word8 patriciaML.ptree

fun mem i = List.exists (fn x => x = i)

(* list of pairs containing initial address and size of a global
 object read from .gar files generated by arm_tracer *)
val global_info = ref []
(* list of (pc,access list) after whole program emulation *)
val access_table = ref []
val ir_access_table = ref []
(* list of read dependencies *)
val rdep_list = ref []
(* list of indexes used in a write instruction and hence relevant *)
val used_index = ref []
(* list of irrelevant indexes *)
val ir_index = ref []
val index = ref 0

fun load_program (s, t) =
   let
      val istrm = TextIO.openIn s
      val a = TextIO.inputAll istrm before TextIO.closeIn istrm
      fun pair [a, b] = (numML.fromHexString a,
                         wordsML.toWord8 (numML.fromHexString b))
        | pair _ = raise Match
   in
      a |> String.tokens (fn c => c = #"\n")
        |> List.map (pair o String.tokens Char.isSpace)
        |> patriciaML.ADD_LIST t
   end

val load_programs = List.foldl load_program patriciaML.Empty

(* ------------------------------------------------------------------------ *)

fun toWord i n = wordsML.fromNum (n, fcpML.ITSELF (numML.fromInt i))
val zero32 = wordsML.toWord32 numML.ZERO
val word8  = wordsML.toWord8 o numML.fromHexString
val word32 = wordsML.toWord32 o numML.fromHexString
fun w2int w = w |> wordsML.w2n |> numML.toInt |> valOf

fun mk_arm_state arch regs psrs md mem =
  armML.arm_state
    (armML.proc numML.ZERO regs,
     armML.proc numML.ZERO psrs,
     combinML.K true,        (* event_register *)
     combinML.K false,       (* interrupt_wait *)
     fn a => case patriciaML.PEEK mem (wordsML.w2n a)
             of SOME d => d
              | NONE => md,
     [],                     (* accesses *)
     armML.ARMinfo (arch, setML.EMPTY, true),
     armML.Coprocessors
       (armML.coproc_state
          (armML.CP14reg zero32,
           armML.CP15reg
            (armML.CP15sctlr (true,  false, false, false, true,  true,
                              false, false, false, false, false, false,
                              false, false, false, false, false, false,
                              true,  false),
             armML.CP15scr   (false, false, false, false, false, false, false),
             armML.CP15nsacr (false, false, false, toWord 14 numML.ZERO),
             zero32, zero32)),
        combinML.K (armML.constC false),
        combinML.K (armML.constC ()),
        combinML.K (armML.constC numML.ZERO),
        combinML.K (combinML.K (armML.constC ())),
        combinML.K (armML.constC []),
        combinML.K (combinML.K (armML.constC ())),
        combinML.K (armML.constC zero32),
        combinML.K (combinML.K (armML.constC ())),
        combinML.K (armML.constC (zero32, zero32))),
     armML.ExclusiveMonitors
       ((combinML.K setML.EMPTY, setML.EMPTY),
        combinML.K (armML.constE ()),
        combinML.K (armML.constE ()),
        combinML.K (armML.constE false),
        combinML.K (armML.constE false),
        combinML.K (armML.constE ()),
        combinML.K (armML.constE ())))

(* ------------------------------------------------------------------------ *)

fun architecture a =
  case a
  of "armv4"   => armML.ARMv4
   | "armv4t"  => armML.ARMv4T
   | "armv5t"  => armML.ARMv5T
   | "armv5te" => armML.ARMv5TE
   | "armv6"   => armML.ARMv6
   | "armv6k"  => armML.ARMv6K
   | "armv6t2" => armML.ARMv6T2
   | "armv7-a" => armML.ARMv7_A
   | "armv7-r" => armML.ARMv7_R
   | _ => raise Fail "architecture"

fun string_of_index i = Int.toString i

fun string_of_rname r =
  case r
  of armML.RName_0usr  => "r0"
   | armML.RName_1usr  => "r1"
   | armML.RName_2usr  => "r2"
   | armML.RName_3usr  => "r3"
   | armML.RName_4usr  => "r4"
   | armML.RName_5usr  => "r5"
   | armML.RName_6usr  => "r6"
   | armML.RName_7usr  => "r7"
   | armML.RName_8usr  => "r8"
   | armML.RName_8fiq  => "r8_fiq"
   | armML.RName_9usr  => "r9"
   | armML.RName_9fiq  => "r9_fiq"
   | armML.RName_10usr => "r10"
   | armML.RName_10fiq => "r10_fiq"
   | armML.RName_11usr => "r11"
   | armML.RName_11fiq => "r11_fiq"
   | armML.RName_12usr => "r12"
   | armML.RName_12fiq => "r12_fiq"
   | armML.RName_SPusr => "sp"
   | armML.RName_SPfiq => "sp_fiq"
   | armML.RName_SPirq => "sp_irq"
   | armML.RName_SPsvc => "sp_svc"
   | armML.RName_SPabt => "sp_abt"
   | armML.RName_SPund => "sp_und"
   | armML.RName_SPmon => "sp_mon"
   | armML.RName_LRusr => "lr"
   | armML.RName_LRfiq => "lr_fiq"
   | armML.RName_LRirq => "lr_irq"
   | armML.RName_LRsvc => "lr_svc"
   | armML.RName_LRabt => "lr_abt"
   | armML.RName_LRund => "lr_und"
   | armML.RName_LRmon => "lr_mon"
   | armML.RName_PC    => "pc"

fun rname i =
  case i
  of 0  => armML.RName_0usr
   | 1  => armML.RName_1usr
   | 2  => armML.RName_2usr
   | 3  => armML.RName_3usr
   | 4  => armML.RName_4usr
   | 5  => armML.RName_5usr
   | 6  => armML.RName_6usr
   | 7  => armML.RName_7usr
   | 8  => armML.RName_8usr
   | 9  => armML.RName_8fiq
   | 10 => armML.RName_9usr
   | 11 => armML.RName_9fiq
   | 12 => armML.RName_10usr
   | 13 => armML.RName_10fiq
   | 14 => armML.RName_11usr
   | 15 => armML.RName_11fiq
   | 16 => armML.RName_12usr
   | 17 => armML.RName_12fiq
   | 18 => armML.RName_SPusr
   | 19 => armML.RName_SPfiq
   | 20 => armML.RName_SPirq
   | 21 => armML.RName_SPsvc
   | 22 => armML.RName_SPabt
   | 23 => armML.RName_SPund
   | 24 => armML.RName_SPmon
   | 25 => armML.RName_LRusr
   | 26 => armML.RName_LRfiq
   | 27 => armML.RName_LRirq
   | 28 => armML.RName_LRsvc
   | 29 => armML.RName_LRabt
   | 30 => armML.RName_LRund
   | 31 => armML.RName_LRmon
   | 32 => armML.RName_PC
   | _ => raise Fail "rname"

fun string_of_psrname p =
  case p
  of armML.CPSR     => "cpsr"
   | armML.SPSR_fiq => "spsr_fiq"
   | armML.SPSR_irq => "spsr_irq"
   | armML.SPSR_svc => "spsr_svc"
   | armML.SPSR_mon => "spsr_mon"
   | armML.SPSR_abt => "spsr_abt"
   | armML.SPSR_und => "spsr_und"

fun psrname i =
  case i
  of 0 => armML.CPSR
   | 1 => armML.SPSR_fiq
   | 2 => armML.SPSR_irq
   | 3 => armML.SPSR_svc
   | 4 => armML.SPSR_mon
   | 5 => armML.SPSR_abt
   | 6 => armML.SPSR_und
   | _ => raise Fail "psrname"

fun encoding armML.Encoding_Thumb   = "16-bit Thumb:\t"
  | encoding armML.Encoding_Thumb2  = "32-bit Thumb:\t"
  | encoding armML.Encoding_ThumbEE = "ThumbEE:\t"
  | encoding armML.Encoding_ARM     = "ARM:\t\t"

fun condition (cond:armML.word4) =
  case w2int cond
  of 0  => "eq"
   | 1  => "ne"
   | 2  => "cs"
   | 3  => "cc"
   | 4  => "mi"
   | 5  => "pl"
   | 6  => "vs"
   | 7  => "vc"
   | 8  => "hi"
   | 9  => "ls"
   | 10 => "ge"
   | 11 => "lt"
   | 12 => "gt"
   | 13 => "le"
   | 14 => "al"
   | _  => raise Fail "condition"

fun data_processing (enc, opc, n) =
  case w2int opc
  of 0  => "and"
   | 1  => "eor"
   | 2  => "sub"
   | 3  => "rsb"
   | 4  => "add"
   | 5  => "adc"
   | 6  => "sbc"
   | 7  => "rsc"
   | 8  => "tst"
   | 9  => "teq"
   | 10 => "cmp"
   | 11 => "cmn"
   | 12 => "orr"
   | 13 => "mov"
   | 14 => "bic"
   | 15 => if enc = armML.Encoding_Thumb2 andalso (w2int n <> 15) then
             "orn"
           else
             "mvn"
   | _  => raise Fail "data_processing"

fun instruction (enc, instr) =
  case instr
  of armML.Unpredictable                      => "unpredictable"
   | armML.Undefined                          => "undefined"
   | armML.Branch (armML.Branch_Target _)     => "branch target"
   | armML.Branch (armML.Branch_Exchange _)   => "branch exchange"
   | armML.Branch (armML.Compare_Branch _)    => "compare branch"
   | armML.Branch (armML.Check_Array _)       => "check array"
   | armML.Branch (armML.Table_Branch_Byte _) => "table branch byte"
   | armML.Branch (armML.Branch_Link_Exchange_Immediate _) =>
       "branch link exchange (imm)"
   | armML.Branch (armML.Branch_Link_Exchange_Register _) =>
       "branch link exchange (reg)"
   | armML.Branch (armML.Handler_Branch_Link _) =>
        "handler branch (link)"
   | armML.Branch (armML.Handler_Branch_Link_Parameter _) =>
        "handler branch with link and parameter"
   | armML.Branch (armML.Handler_Branch_Parameter _) =>
        "handler branch with parameter"
   | armML.DataProcessing (armML.Data_Processing (opc, _, n, _, _)) =>
       data_processing (enc, opc, n)
   | armML.DataProcessing (armML.Add_Sub (add, _, _, _)) =>
       if add then "add (wide)" else "sub (wide)"
   | armML.DataProcessing (armML.Move_Halfword _)     => "move halfword"
   | armML.DataProcessing (armML.Divide _)            => "divide"
   | armML.DataProcessing (armML.Multiply _)          => "multiply"
   | armML.DataProcessing (armML.Multiply_Subtract _) => "multiply subtract"
   | armML.DataProcessing (armML.Multiply_Long _)     => "multiply (long)"
   | armML.DataProcessing (armML.Saturate _)          => "saturate"
   | armML.DataProcessing (armML.Saturate_16 _)       => "saturate (16)"
   | armML.DataProcessing (armML.Select_Bytes _)      => "select bytes"
   | armML.DataProcessing (armML.Extend_Byte _)       => "extend byte"
   | armML.DataProcessing (armML.Extend_Byte_16 _)    => "extend byte (16)"
   | armML.DataProcessing (armML.Extend_Halfword _)   => "extend halfword"
   | armML.DataProcessing (armML.Pack_Halfword _)     => "pack halfword"
   | armML.DataProcessing (armML.Reverse_Bits _)      => "reverse bits"
   | armML.DataProcessing (armML.Byte_Reverse_Word _) => "byte reverse word"
   | armML.DataProcessing (armML.Byte_Reverse_Packed_Halfword _) =>
       "byte reverse packed halfword"
   | armML.DataProcessing (armML.Byte_Reverse_Signed_Halfword _) =>
       "byte reverse signed halfword"
   | armML.DataProcessing (armML.Signed_Halfword_Multiply _) =>
       "signed halfword multiply"
   | armML.DataProcessing (armML.Signed_Multiply_Dual _) =>
       "signed multiply dual"
   | armML.DataProcessing (armML.Signed_Multiply_Long_Dual _) =>
       "signed multiply dual (long)"
   | armML.DataProcessing (armML.Signed_Most_Significant_Multiply _) =>
       "signed most significant multiply"
   | armML.DataProcessing (armML.Signed_Most_Significant_Multiply_Subtract _) =>
       "signed most significant multiply subtract"
   | armML.DataProcessing (armML.Multiply_Accumulate_Accumulate _) =>
       "multiply accumulate accumulate"
   | armML.DataProcessing (armML.Saturating_Add_Subtract _) =>
       "saturating add subtract"
   | armML.DataProcessing (armML.Bit_Field_Clear_Insert _) =>
       "bit field clear/insert"
   | armML.DataProcessing (armML.Bit_Field_Extract _) =>
       "bit field extract"
   | armML.DataProcessing (armML.Count_Leading_Zeroes _) =>
       "count leading zeroes"
   | armML.DataProcessing (armML.Unsigned_Sum_Absolute_Differences _) =>
       "unsigned sum absolute differences"
   | armML.DataProcessing (armML.Parallel_Add_Subtract _) =>
       "parallel add subtract"
   | armML.StatusAccess (armML.Status_to_Register _) =>
       "program status to register (mrs)"
   | armML.StatusAccess (armML.Register_to_Status _) =>
       "register to program status (msr)"
   | armML.StatusAccess (armML.Immediate_to_Status _) =>
       "immediate to program status (msr)"
   | armML.StatusAccess (armML.Change_Processor_State _) =>
       "change processor state"
   | armML.StatusAccess (armML.Set_Endianness _) =>
       "set endianess"
   | armML.LoadStore (armML.Load _)            => "load"
   | armML.LoadStore (armML.Store _)           => "store"
   | armML.LoadStore (armML.Load_Halfword _)   => "load halfword"
   | armML.LoadStore (armML.Store_Halfword _)  => "store halfword"
   | armML.LoadStore (armML.Load_Dual _)       => "load dual"
   | armML.LoadStore (armML.Store_Dual _)      => "store dual"
   | armML.LoadStore (armML.Load_Multiple _)   => "load multiple"
   | armML.LoadStore (armML.Store_Multiple _)  => "store multiple"
   | armML.LoadStore (armML.Load_Exclusive _)  => "load exclusive"
   | armML.LoadStore (armML.Store_Exclusive _) => "store exclusive"
   | armML.LoadStore (armML.Load_Exclusive_Doubleword _) =>
       "load exclusive doubleword"
   | armML.LoadStore (armML.Store_Exclusive_Doubleword _) =>
       "store exclusive doubleword"
   | armML.LoadStore (armML.Load_Exclusive_Halfword _) =>
       "load exclusive halfword"
   | armML.LoadStore (armML.Store_Exclusive_Halfword _) =>
       "store exclusive halfword"
   | armML.LoadStore (armML.Load_Exclusive_Byte _) =>
       "load exclusive byte"
   | armML.LoadStore (armML.Store_Exclusive_Byte _) =>
       "store exclusive byte"
   | armML.LoadStore (armML.Store_Return_State _) =>
       "store return state"
   | armML.LoadStore (armML.Return_From_Exception _) =>
       "return from exception"
   | armML.Miscellaneous armML.Clear_Exclusive         => "clear exclusive"
   | armML.Miscellaneous (armML.Hint _)                => "hint"
   | armML.Miscellaneous (armML.Breakpoint _)          => "breakpoint"
   | armML.Miscellaneous (armML.Swap _)                => "swap"
   | armML.Miscellaneous (armML.Preload_Data _)        => "preload data"
   | armML.Miscellaneous (armML.Preload_Instruction _) => "preload instruction"
   | armML.Miscellaneous (armML.Supervisor_Call _)     => "supervisor call"
   | armML.Miscellaneous (armML.Secure_Monitor_Call _) => "secure monitor call"
   | armML.Miscellaneous (armML.If_Then _)             => "if-then"
   | armML.Miscellaneous (armML.Enterx_Leavex true)    => "enterx"
   | armML.Miscellaneous (armML.Enterx_Leavex false)   => "leavex"
   | armML.Miscellaneous (armML.Data_Memory_Barrier _) => "data memory barrier"
   | armML.Miscellaneous (armML.Data_Synchronization_Barrier _) =>
       "data synchronization barrier"
   | armML.Miscellaneous (armML.Instruction_Synchronization_Barrier _) =>
       "instruction synchronization barrier"
   | armML.Coprocessor _    => "coprocessor"

fun for_se base top f =
  let
     fun For i = if i > top then () else (f i; For (i+1))
  in
     For base
  end

fun for_word32 base top f =
   let
      val t = word32 top
      val b = word32 base
      val one = wordsML.toWord32 numML.ONE
      val add1 = wordsML.word_add one
      fun For i = if wordsML.word_gt i t then () else (f i; For (add1 i))
   in
      For b
   end

fun hex n s = StringCvt.padLeft #"0" n (wordsML.word_to_hex_string s)

val traceOut = ref TextIO.stdOut

fun out l = TextIO.output (!traceOut, String.concat (l @ ["\n"]))

fun out' l = TextIO.output (!traceOut, String.concat (l))

fun deps dl =
  List.foldr
  (fn ((d,d_r),acm) =>
    set_union acm d_r
  ) [] dl

fun ir_indexes il wl = set_diff il wl

fun read_global_info gar =
    let
        val istrm = TextIO.openIn gar
        val a = TextIO.inputAll istrm before TextIO.closeIn istrm
        fun pair [a, b, c] = (wordsML.fromString32 a,wordsML.fromString32 b)
          | pair _ = raise Match
    in
        (
          a |> String.tokens (fn c => c = #"\n")
            |> List.map (pair o String.tokens Char.isSpace)
        )
    end

fun print_global_info () =
    (out ["= GLOBAL INFO = "];
     List.app
         (fn (a,b) => out ["= ",wordsML.word_to_hex_string a," ",wordsML.word_to_hex_string b]
         ) (! global_info)
    )

fun is_global_address addr =
    List.exists
    (fn (init,size) =>
        (wordsML.word_ge addr init) andalso (wordsML.word_lt addr (wordsML.word_add init size))
    ) (! global_info)

fun is_stack_address addr =
    String.isPrefix "F" (wordsML.word_to_hex_string addr)

fun mem_to_index mem i =
    case mem
    of armML.MEM_READ (a,vl,s) => if is_stack_address a then armML.LOCAL_READ a
                              else armML.GLOBAL_READ i
    | armML.MEM_WRITE (a,vl,s) => if is_stack_address a then armML.LOCAL_WRITE a
                              else armML.GLOBAL_WRITE i

fun convert_access_to_ir_access acc_table =
  List.map
  (fn (pc,acc_list) =>
     let val new_acc_list = List.map
     (fn acc =>
       case acc
       of armML.MEM_READ _ => (index := !index+1;mem_to_index acc (! index))
       | armML.MEM_WRITE _ => (index := !index+1;mem_to_index acc (! index))
       | armML.REG_READ r  => armML.IR_REG_READ r
       | armML.REG_WRITE r => armML.IR_REG_WRITE r
     ) acc_list
     in
        (pc,new_acc_list)
     end
  ) acc_table

fun read_set s =
  List.filter
  (fn x =>
    case x
    of armML.IR_REG_READ armML.RName_PC => false
    | armML.IR_REG_READ _ => true
    | armML.LOCAL_READ _ => true
    | armML.GLOBAL_READ _=> true
    | _  => false
  ) s

fun write_set s =
    List.filter
        (fn x =>
            case x
             of armML.IR_REG_WRITE armML.RName_PC => false
              | armML.IR_REG_WRITE _ => true
              | armML.LOCAL_WRITE _ => true
              | armML.GLOBAL_WRITE _=> true
              | _  => false
        ) s

fun access_list s = List.rev (armML.arm_state_accesses s)

fun print_ir_access_list s =
    List.app
        (fn acc =>
            (case acc of armML.IR_REG_WRITE r      => out' [string_of_rname r]
               | armML.IR_REG_READ r       => out' [string_of_rname r]
               | armML.LOCAL_READ i     => out' ["L",wordsML.word_to_hex_string i]
               | armML.LOCAL_WRITE i    => out' ["L",wordsML.word_to_hex_string i]
               | armML.GLOBAL_READ i    => out' ["G",string_of_index i]
               | armML.GLOBAL_WRITE i   => out' ["G",string_of_index i]
            ;print " ")
        ) s

fun print_ir_indexes ir_list = (
    print "=== IR Indexes ===\nIR : ";
    print_ir_access_list ir_list;
    print "\n"
    )

fun print_index msg wl = (
    print msg;
    print_ir_access_list wl;
    print "\n"
)

fun dump_ir_access_table s=
    let val _ = out ["=== Access Table Dump ==="]
    in
    List.app
         (fn elem =>
             case elem of (pc,acc_list) =>
                          (out ["= PC :: ", hex 8 pc];
          print_ir_access_list acc_list;print "\n")

         ) s
    end

fun print_dependency_list dl =
    (print "= DEPLIST : {";
     List.app
         (fn dep => case dep of (d,ds) =>
                                (print "(";print_ir_access_list [d]; out' [" : "];
                                 print_ir_access_list ds;print ") ; ")
         ) dl;
    print " }\n"
    )

(* fun find_deps dl r = *)
(*     case dl *)
(*     of [] => [] *)
(*     | (r',d_r')::xs => if r = r' then (print "= found";d_r') else find_deps xs r *)

fun find_deps dl r =
    let val res = List.find (fn (r',d_r') => armML.compare_ir_access r r') dl
    in
    case res
    of SOME (r,d_r) => d_r
    | NONE => []
    end

fun add_deps dl r d_r =
    let
      fun foo dl r d_r =
        case dl
        of [] => [(r,d_r)]
        | (r',old_d_r')::xs => if armML.compare_ir_access r r' then (r',set_union old_d_r' d_r)::xs else (r',old_d_r')::foo xs r d_r
    in
      rdep_list := foo dl r d_r
    end

fun empty_deps dl r =
    let
        fun foo dl r =
            case dl
             of [] => []
              | (r',old_d_r')::xs => if armML.compare_ir_access r r' then (r',[])::xs else (r',old_d_r')::foo xs r
    in
        rdep_list := foo dl r
    end

(* fun rdep_minus_wdep rdl wdl = *)
(*     List.map *)
(*         (fn (d,d_r) => *)
(*             (d,set_diff d_r (find_deps wdl d)) *)
(*         ) rdl *)

(* fun rdep_minus_wdep rdl wdl = *)
(*     let val irdl = ir_indexes rdl *)
(*         fun foo irdl rdl wdl = *)
(*             case rdl *)
(*             of [] => irdl *)
(*             | (r,d_r)::rxs => (case (find_deps wdl r) *)
(*                  of [] => foo irdl rxs wdl *)
(*                  | d_w => foo (set_diff irdl (find_deps rdl r)) rxs wdl) *)
(*     in *)
(*         foo irdl rdl wdl *)
(*     end *)

fun update_deps dl w r =
    case r
      of armML.IR_REG_READ _ => (case w
                             of armML.IR_REG_WRITE _ => add_deps dl w (find_deps dl r)
                             | armML.GLOBAL_WRITE _ => used_index := set_union (! used_index) (find_deps dl r)
                             | armML.LOCAL_WRITE _  => add_deps dl w (find_deps dl r) )
      | armML.LOCAL_READ _ => (case w
                               of armML.IR_REG_WRITE _ => add_deps dl w (find_deps dl r)
                               | armML.GLOBAL_WRITE _ => used_index := set_union (! used_index) (find_deps dl r)
                               | armML.LOCAL_WRITE _  => add_deps dl w (find_deps dl r) )
      | armML.GLOBAL_READ _ => (case w
                                 of armML.IR_REG_WRITE _ => add_deps dl w [r]
                                  | armML.GLOBAL_WRITE _ => used_index := set_union (! used_index) (find_deps dl r)
                                  | armML.LOCAL_WRITE _  => add_deps dl w [r] )

fun readSet_writeSet_list frs fws acc_table =
    List.map
    (fn (pc,acc_list) => (pc,frs acc_list,fws acc_list)
    ) acc_table

fun print_readSet_writeSet_list s =
    let val _ = out ["=== ReadSet WriteSet Dump ==="]
        in
            List.app
                (fn elem =>
                    case elem of (pc,rs,ws) =>
                                 (out ["= PC :: ", hex 8 pc];

                                  out' ["= ReadSet : "];
                                  print_ir_access_list rs;print "\n";
                                  out' ["= WriteSet : "];
                                  print_ir_access_list ws;print "\n")

                ) s
        end


fun create_read_dependency_list frs fws dl acc_table =
        List.app
        (fn (pc,rs,ws) =>
         (* (out ["= PC :: ", hex 8 pc]; *)
         (* out' ["= ReadSet : "]; *)
         (* print_ir_access_list rs;print "\n"; *)
         (* out' ["= WriteSet : "]; *)
         (* print_ir_access_list ws;print "\n"; *)
         List.app
          (fn w =>
            ((case w
            of armML.IR_REG_WRITE _ =>
              if List.exists
                 (fn r => armML.compare_ir_access w r) rs
              then ()
              else empty_deps (! dl) w
            | armML.LOCAL_WRITE _ =>
              if List.exists
                (fn r => armML.compare_ir_access w r) rs
              then ()
              else empty_deps (! dl) w
            | _ => ());
            List.app
            (fn r =>
              (update_deps (! dl) w r;
              ir_index := set_union (! ir_index) (deps (! dl)) )
            ) rs)
          ) ws
          (* ;print_dependency_list (! rdep_list);print_index "= USED INDEX : " (! used_index)) *)
         ) (readSet_writeSet_list frs fws acc_table)

fun print_trace s =
   List.app
     (fn acc =>
           case acc of
             armML.MEM_WRITE (a, vl, s)=> out (["Store : ", hex 8 a, " : "] @ (List.map (hex 2) vl) @ [" : ", (numML.toDecString s)])
           | armML.MEM_READ (a, vl, s) => out (["Load : ", hex 8 a, " : "] @ (List.map (hex 2) vl) @ [" : ", (numML.toDecString s)])
           | _                      => ()
          (* | armML.REG_WRITE r      => out ["= w: ", string_of_rname r] *)
          (* | armML.PSR_WRITE r      => out ["= w: ", string_of_psrname r] *)
          (* | armML.REG_READ r       => out ["= r: ", string_of_rname r] *)
          (* | armML.PSR_READ r       => out ["= r: ", string_of_psrname r] *)
    )
    (access_list s)

fun print_arm_state s =
   let
      val reg = armML.arm_state_registers s
      val psr = armML.arm_state_psrs s
      val ii = armML.iiid numML.ZERO
      val id = armML.iiid_proc ii
      fun pad n s = StringCvt.padRight #" " n s ^ ": "
   in
      out ["= General Purpose Registers\n",
           "===========================\n"];
      for_se 0 32 (fn i =>
        let
           val r = rname i
        in
           out ["= ", pad 9 (string_of_rname r), hex 8 (reg (id, r))]
        end);
      out ["\n= Program Status Registers\n",
           "==========================\n"];
      for_se 0 6 (fn i =>
        let
           val r = psrname i
        in
           out ["= ", pad 9 (string_of_psrname r),
                hex 8 (armML.encode_psr (psr (id, r)))]
        end)
   end

local
   fun print_line (a, b : wordsML.word8) =
      out ["= [", hex 8 (wordsML.toWord32 a), "] : ", hex 2 b]
   fun mem_compare ((a1, _), (a2, _)) =
      if a1 = a2 then
         General.EQUAL
      else if numML.< a1 a2 then
         General.LESS
      else
         General.GREATER
   val print_mem = List.app print_line o msort mem_compare
in
   val print_arm_mem = print_mem o patriciaML.toList
   fun print_diff_arm_mem prog1 prog2 =
   (* FZ: this is awfully slow, commenting it out *)
(*   let
      val new = set_diff (patriciaML.toList prog2) (patriciaML.toList prog1)
   in *)
      out ["\n= Modified Memory\n",
           "=================\n",
	   "= SKIPPED"]
(*      print_mem new *)
(*   end *)
end

fun print_arm_run prog (message, prog_state) =
  (if message = "" then () else
     out ["= Final Message\n", "===============\n\n+ ", message, "\n"];
   case prog_state
   of SOME (prog', state) =>
        (print_arm_state state; print_diff_arm_mem prog prog')
   | _ => out ["state upredictable"])

(* ------------------------------------------------------------------------ *)

fun update_prog p [] = p
  | update_prog p (armML.MEM_WRITE (a, [d], s) :: l) =
      update_prog (patriciaML.ADD p (wordsML.w2n a, d)) l
  | update_prog p (armML.MEM_WRITE (a, dl, s) :: l) =
      let fun unroll a [] = []
            | unroll a (d::dl) =
               (armML.MEM_WRITE (a, [d], numML.ONE)) ::
               (unroll (wordsML.word_add a (wordsML.n2w_itself (numML.ONE,(fcpML.ITSELF(numML.fromString "32"))))) dl)
      in
        update_prog p ((unroll a (List.rev dl)) @ l)
      end
  | update_prog p (_ :: l) = update_prog p l

fun ptree_arm_run (prog, state) t (lock_addr,unlock_addr)=
   let
      val ii = armML.iiid numML.ZERO
      val arch = case armML.read_arch ii state
                 of armML.Error s => raise Fail "couldn't read Architecture"
                  | armML.ValueState (a, _) => a
      fun ptree_arm_loop prog' cycle t (lock_addr,unlock_addr) =
        armML.seqT (armML.waiting_for_interrupt ii) (fn wfi =>
          if wfi orelse t = 0 then
            armML.constT ((prog', cycle), "finished")
          else
            armML.seqT (armML.read_pc ii) (fn pc =>
            if hex 8 pc = lock_addr  then
                armML.seqT (armML.handle_locks ii armML.Lock) (fn _ =>
              ptree_arm_loop prog' (cycle+1) (if t < 0 then t else t - 1) (lock_addr,unlock_addr))
            else if hex 8 pc = unlock_addr then
                armML.seqT (armML.handle_locks ii armML.Unlock) (fn _ =>
              ptree_arm_loop prog' (cycle+1) (if t < 0 then t else t - 1) (lock_addr,unlock_addr))
            else
            armML.attempt (prog', cycle)
              (armML.fetch_instruction ii
                 (armML.ptree_read_word prog')
                 (armML.ptree_read_halfword prog'))
              (fn instr =>
                 armML.seqT
                   (armML.writeT
                      (armML.arm_state_accesses_fupd (combinML.K [])))
                        (fn _ =>
                          armML.seqT (armML.readT combinML.I) (fn s1 =>
                            armML.seqT (armML.arm_instr ii (pairML.SND instr))
                              (fn _ => armML.seqT (armML.readT combinML.I)
                                (fn s2 =>
                                    let
                                    val pc = armML.arm_state_registers s1
                                                 (numML.ZERO, armML.RName_PC)
                                    (* val _ = out ["= pc :: ",hex 8 pc] *)
                                    (* val _ = out ["= Read Set :"] *)
                                    (* val _ = print_access_list (read_set s2) *)
                                    (* val _ = out ["= Write Set :"] *)
                                    (* val _ = print_access_list (write_set s2) *)
                                    val _ = print_trace s2
                                    in
                                       (access_table := !access_table @ [(pc,access_list s2)];
                                       ptree_arm_loop
                                         (update_prog prog'
                                            (armML.arm_state_accesses s2))
                                         (cycle + 1)
                                         (if t < 0 then t else t - 1) (lock_addr,unlock_addr))
                                    end)))))))
   in
      case ptree_arm_loop prog 0 t (lock_addr,unlock_addr) state
      of armML.Error e => (e, NONE)
       | armML.ValueState (((prog', c), v), s') =>
           ("at cycle " ^ Int.toString c ^ ": " ^ v, SOME (prog', s'))
   end

(* ------------------------------------------------------------------------ *)

val lower_string = String.implode o map Char.toLower o String.explode

fun pluck P =
   let fun pl _ [] = raise Fail "pluck: predicate not satisfied"
         | pl A (h::t) = if P h then (h, List.revAppend(A, t)) else pl (h::A) t
   in
      pl []
   end

fun init_config prog s =
   let
      val l =
         s |> String.tokens (fn c => mem c [#",", #";",#"\n"])
           |> map (fn a =>
                   case String.tokens (fn c => c = #"=" orelse Char.isSpace c) a
                     of [l, r] => (lower_string l, lower_string r)
                      | _ => raise Fail "init_config")
      val ((_, arch), l) = pluck (fn (n, _) => n = "arch") l
                           handle Fail _ => (("", "armv7-a"), l)
      val ((_, default_reg), l) = pluck (fn (n, _) => n = "reg_") l
                                  handle Fail _ => (("", "0"), l)
      val ((_, default_psr), l) = pluck (fn (n, _) => n = "_psr") l
                                  handle Fail _ => (("", "10"), l)
      val ((_, default_mem), l) = pluck (fn (n, _) => n = "mem_") l
                                  handle Fail _ => (("", "0"), l)
      val dreg = word32 default_reg
      val dpsr = armML.decode_psr (word32 default_psr)
   in
      mk_arm_state (architecture arch)
       (fn r => case List.find (fn (n, _) => string_of_rname r = n) l
                of SOME (_, v) => word32 v
                 | _ => dreg)
       (fn r => case List.find (fn (n, _) => string_of_psrname r = n) l
                of SOME (_, v) => armML.decode_psr (word32 v)
                 | _ => dpsr)
       (word8 default_mem) prog
   end

(* ------------------------------------------------------------------------ *)

fun time f x =
  let
    fun p t =
      let
        val s = Time.fmt 3 t
      in
        case size (List.last (String.fields (fn x => x = #".") s)) of 3 => s
        | 2 => s ^ "0"
        | 1 => s ^ "00"
        | _ => raise Fail "time"
      end
    val c = Timer.startCPUTimer ()
    val r = Timer.startRealTimer ()
    fun pt () =
      let
        val {usr, sys, ...} = Timer.checkCPUTimer c
        val real = Timer.checkRealTimer r
      in
        print
        ("= User: " ^ p usr ^ "  System: " ^ p sys ^ "  Real: " ^ p real ^ "\n")
      end
    val y = f x handle e => (pt (); raise e)
    val () = pt ()
  in
    y
  end

fun input_number P message =
   let
      val _ = print message
   in
      case TextIO.inputLine TextIO.stdIn
      of NONE => input_number P message
       | SOME s => case Int.fromString s
                   of SOME n => if P n then n else input_number P message
                    | NONE => input_number P message
   end

fun arm_run prog options count m=
   let
      val state = init_config prog options
   in
      ptree_arm_run (prog, state) count m
   end

fun nth_or_empty ls n =
        if List.length ls > n then List.nth (ls,n) else ""

fun safe_take ls n =
        if List.length ls >= n then List.take (ls,n) else ls

in

fun main () =
   let
      val args = CommandLine.arguments()
      val filename = List.hd (safe_take args 1)
      val gar =  String.concat [String.substring (filename,0,((String.size filename) - 6)),".gar"]
      val _ = global_info := read_global_info gar
      val prog = load_programs (safe_take args 1)
      val lock_addr = nth_or_empty args 1
      val unlock_addr = nth_or_empty args 2
      val _ = print "= Enter architecture, initial register values and default\
                    \ memory content.\n= (Enter values as Hex.)\n\
                    \= For example: arch = ARMv7-A, pc = A00, r0 = AF, r_ = 10,\n= \
                    \             cpsr = 80000010, _psr = 10, mem_ = 0\n= >\n"
      val options = valOf (TextIO.inputLine TextIO.stdIn)
      val count = input_number (fn i => ~1 <= i) "= Enter number of cycles:\n"
   in
      case time (arm_run prog options) count (lock_addr,unlock_addr)
      of out as (_, SOME _) =>
        (print_arm_run prog out;
         ir_access_table := convert_access_to_ir_access (! access_table);
         create_read_dependency_list read_set write_set rdep_list (! ir_access_table);
         print_dependency_list (! rdep_list);
         print_index "= USED INDEX : " (! used_index);
         print_index "= DEP INDEX  : " (! ir_index);
         print_ir_indexes (ir_indexes (! ir_index) (! used_index))
        )
       | (e, NONE) => out [e]
   end

end

val () = PolyML.shareCommonData main;
val () = PolyML.export("HOLarm", main)

(*
val prog = load_programs ["md5.o"];
val options = "pc = 8000, r0 = C0000, lr = A0000, sp = B0000, cpsr = 10";

gcc -o run HOLarm.o -L$HOME/poly/lib -lpolymain -lpolyml

On MacOS:
gcc -o run HOLarm.o -L$HOME/poly/lib -lpolymain -lpolyml -segprot POLY rwx rwx

*)
