(**************************************************************************)
(*                            arm-run & arm-tracer                        *)
(*                                                                        *)
(*   Pankaj More, IIT Kanpur & INRIA Paris-Rocquencourt                   *)
(*   Francesco Zappa Nardelli, INRIA Paris-Rocquencourt                   *)
(*                                                                        *)
(*  The arm-run and arm-tracer tools are copyright 2012, 2013 Institut    *)
(*  National de Recherche en Informatique et en Automatique (INRIA).      *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(*  *******************************************************************   *)
(*                                                                        *)
(*  *******************************************************************   *)
(*                                                                        *)
(*                               arm-run                                  *)
(*                                                                        *)
(*  The arm-run tool is copyright 2010 - 2013 Anthony Fox, Magnus Myreen  *)
(*  and Mike Gordon, Computer Laboratory, University of Cambridge.        *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(*                                                                        *)
(**************************************************************************)

(* options *)

let debug = ref false
let trace_u_val = ref false
let index_map = ref []
let index = ref 0
let trace_index = ref 0

(* util *)
let strip_zeros s =
  let rec count i = match s.[i] with
    | '0' -> count (i+1)
    | _   -> i
  in
  let num = count 0 in
  String.sub s num ((String.length s)-num)

let strip ?(chars=" \t\r\n") s =
  let p = ref 0 in
  let l = String.length s in
  while !p < l && String.contains chars s.[!p] do
    incr p;
  done;
  let p = !p in
  let l = ref (l - 1) in
  while !l >= p && String.contains chars s.[!l] do
    decr l;
  done;
  String.sub s p (!l - p + 1)

let exec c =
  if !debug then print_endline ("exec: "^c);
  Sys.command c

let error s =
  print_endline ("error: "^s);
  exit 1

(* options *)

let base_address = 0x0

(* parse *)
let is_mem_word v =
  try
    int_of_string ("0x"^(strip v));
    if String.length (strip v) = 8 then true else false
  with _ -> false

let rec take_while f l =
  match l with
  | [] -> []
  | x :: xs ->
    match f x with
    | true -> x :: take_while f xs
    | false -> []

let convert_dump dump output =
  let fd_in =
    try open_in dump with Sys_error _ -> error ("can't open " ^ dump) in
  let fd_out =
    try open_out output with Sys_error _ -> error ("can't open " ^ output) in
  let rec parse_line () =
    try
      let line = input_line fd_in in
      try
	if line.[0] != ' ' then parse_line () else
          let l = Str.split (Str.regexp ":") line in
          let addr = ref (int_of_string ("0x"^(strip (List.hd l)))) in
          let values = take_while is_mem_word
            (List.map (fun x-> String.uppercase (strip x))
               (Str.split (Str.regexp " ") (List.hd (List.tl l)))) in
          List.iter
            (fun value ->
              Printf.fprintf fd_out "%04X %s\n" (base_address+ !addr) (String.sub value 6 2);incr addr;
              Printf.fprintf fd_out "%04X %s\n" (base_address+ !addr) (String.sub value 4 2);incr addr;
              Printf.fprintf fd_out "%04X %s\n" (base_address+ !addr) (String.sub value 2 2);incr addr;
              Printf.fprintf fd_out "%04X %s\n" (base_address+ !addr) (String.sub value 0 2);incr addr;
            ) values;
	  parse_line ()
      with Invalid_argument _ -> parse_line ()
    with End_of_file -> close_in fd_in; close_out fd_out in
  parse_line ()

let find_main_vars dump =
  let pc_main = ref "-1" in
  let vars = ref [] in
  let fd_in =
    try open_in dump with Sys_error _ -> error ("can't open " ^ dump) in
  let rgx = Str.regexp "<.+>" in
  let rec parse_line () =
    try
      let line = input_line fd_in in
      try
	if line.[0] = ' ' then parse_line () else begin
	  try
	    let _ = Str.search_forward rgx line 0 in
            let mtc = Str.matched_string line in
            let var_name = String.sub mtc 1 ((String.length mtc)-2) in
            if (var_name.[0] = 'g'  || var_name.[0] = 'c' || var_name.[0] = 'a')
	(*	&& var_name.[1] = '_') *)
	      || var_name = "main" then begin
              let addr_idx = String.index line ' ' in
              let address = String.uppercase (strip_zeros (String.sub line 0 addr_idx)) in
	      if !debug then Printf.printf "* variable: %s at %s\n" var_name address;
	      if var_name = "main"
	      then pc_main := address
	      else vars := (address,var_name)::!vars
	    end;
	    parse_line ()
	  with Not_found -> parse_line ()
	end
      with Invalid_argument _ -> parse_line ()
    with End_of_file -> close_in fd_in in
  parse_line ();
  if !pc_main = "-1" then error "cannot read the address of main";
  (!pc_main,!vars)

let convert_to_hex s =
  if Str.string_match (Str.regexp "0x") s 0
  then int_of_string s
  else int_of_string ("0x" ^ (strip s))

let list_global_addresses elf gar =
  let fd_in =
    try open_in elf with Sys_error _ -> error ("can't open " ^ elf) in
  let fd_out =
    try open_out gar with Sys_error _ -> error ("can't open " ^ gar) in
  let rgx = Str.regexp ".+OBJECT.+GLOBAL.+DEFAULT" in
  let addrs = ref [] in
  let rec parse_line () =
    try
      let line = input_line fd_in in
      try
        let _ = Str.search_forward rgx line 0 in
        let parse = Str.split (Str.regexp " +") line in
        (* List.iter (fun s -> print_string s; print_string "\t") parse; *)
        (* print_int (convert_to_hex(List.nth parse 1)); *)
        (* print_string "\n"; *)
        let element = ((convert_to_hex(List.nth parse 1)), int_of_string(List.nth parse 2), List.nth parse 7) in
        addrs := element::!addrs;
        output_string fd_out (string_of_int(convert_to_hex(List.nth parse 1)) ^ " " ^ (List.nth parse 2) ^ " " ^ (List.nth parse 7) ^ "\n");
        parse_line ()
      with Not_found -> parse_line ()
    with End_of_file -> close_in fd_in;close_out fd_out  in
  parse_line ();
  !addrs

let find_pthread_mutex_calls elf rgxv=
  let fd_in =
    try open_in elf with Sys_error _ -> error ("can't open " ^ elf) in
  let rgx = Str.regexp rgxv in
  let addr = ref "" in
  let rec parse_line () =
    try
      let line = input_line fd_in in
      try
        let _ = Str.search_forward rgx line 0 in
        let parse = Str.split (Str.regexp " +") line in
        (* List.iter (fun s -> print_string s; print_string "\t") parse; *)
        (* print_int (convert_to_hex(List.nth parse 1)); *)
        (* print_string "\n"; *)
        let element = String.uppercase(List.nth parse 1) in
        addr := element;
      with Not_found -> parse_line ()
    with End_of_file -> close_in fd_in in
  parse_line ();
  !addr

let is_global_access addresses line =
  let [op;addr;value;size] = Str.split (Str.regexp ":") line in
  let hexaddr = convert_to_hex addr in
  List.exists (fun (init,size) -> (hexaddr >= init && hexaddr < init + size)) addresses

(* annotate trace entry *)
let is_stack_accesses line =
  let [op;addr;value;size] = Str.split (Str.regexp ":") line in
  let addr_strip = try strip_zeros (strip addr) with _ -> "0" in
  addr_strip.[0] = 'F'

let int_of_ir ir = int_of_string (String.sub ir 1 (String.length ir - 1))

let ir_indexes_from_arm_run line =
  let ["IR ";ls] = Str.split (Str.regexp ":") line in
  let ir = Str.split (Str.regexp " ") ls in
  List.map int_of_ir
    (List.filter
       (fun i ->
         not (Str.first_chars i 2 = "LF")
       )
    ir)

let rewrite_indexes index_map ir_indexes =
  let filtered_indexes = List.filter (fun i -> List.exists (fun (x,y) -> i = x ) index_map) ir_indexes in
  List.map
  (fun i ->
    snd (List.find (fun (x,y) -> i = x) index_map)
  ) filtered_indexes

let show_list ls =
  String.concat " " (List.map string_of_int ls)

(* main *)

let _ =
  let completed = ref false in
  let file = ref "" in
  let options =
    Arg.align
      [("-trace_indirect",
	Arg.Unit (fun () -> trace_u_val := true),
	" trace the accesses that load addresses from .text");
       ("-debug",
	Arg.Unit (fun () -> debug := true),
	" debug")] in

  Arg.parse options
    (fun s -> if !file = "" then file := s else error "too many files")
    "use: arm-tracer [options] <file>";
  let objdump_name = !file ^ ".dump" in
  let elf_name = !file ^ ".elf" in
  let gar_name = !file ^ ".gar" in
  let output_name = !file ^ ".bytes" in
  let trace_name = !file ^ ".log" in
  ignore (exec ("arm-none-linux-gnueabi-objdump -d -z -j .text -j .data -j .rodata -j .bss "
		^ !file^" > "^objdump_name));
  ignore (exec ("arm-none-linux-gnueabi-readelf -s " ^ !file^" > "^elf_name));
  convert_dump objdump_name output_name;
  let (pc_main,vars) = find_main_vars objdump_name in
  let lock_addr = find_pthread_mutex_calls elf_name "FUNC.+GLOBAL.+DEFAULT.+UND.+pthread_mutex_lock" in
  let unlock_addr = find_pthread_mutex_calls elf_name "FUNC.+GLOBAL.+DEFAULT.+UND.+pthread_mutex_unlock" in
  flush stdout;
  (* protocol to drive the Arm emulator *)
  if !debug then
    Printf.printf "\n*** Calling: arm_run %s %s %s\n\n" output_name lock_addr unlock_addr;
  let addresses = List.map (fun (x,y,z) -> (x,y)) (list_global_addresses elf_name gar_name) in
  let (fdin,fdout) = Unix.open_process ("arm_run "^output_name^" "^lock_addr^" "^unlock_addr) in
  (* discard the initial message *)
  if !debug then begin
    Printf.printf "< %s\n" (input_line fdin);
    Printf.printf "< %s\n" (input_line fdin);
    Printf.printf "< %s\n" (input_line fdin);
    Printf.printf "< %s\n" (input_line fdin);
    Printf.printf "< %s\n" (input_line fdin);
  end else begin
    ignore (input_line fdin);
    ignore (input_line fdin);
    ignore (input_line fdin);
    ignore (input_line fdin);
    ignore (input_line fdin);
  end;
  (* set the pc *)
  let pc_string = "pc = "^pc_main^"\n" in
  if !debug then
    Printf.printf "> %s\n" pc_string;
  output_string fdout pc_string;
  flush fdout;
  (* set the number of cycles *)
  if !debug then
    Printf.printf "< %s\n" (input_line fdin)
  else
    ignore (input_line fdin);
  let cycle_string = "10000\n" in
  if !debug then
    Printf.printf "> %s\n" cycle_string;
  output_string fdout cycle_string;
  flush fdout;
  let fd_trace_out =
    try open_out trace_name
    with _ -> error ("can't open "^trace_name)
  in
  output_string fd_trace_out "===========================\n";
  output_string fd_trace_out "= Generated by arm_tracer =\n";
  output_string fd_trace_out "===========================\n";
  try
    while (true) do
      let l = input_line fdin in
      if !debug then print_endline ("< "^l) else ();
      if l = "" then ()
      else if l.[0] = '=' then ()
      else if l.[0] = '+' then
	let rgx = Str.regexp_string "couldn't fetch an instruction" in
	if (try ignore (Str.search_forward rgx l 0); true
	  with Not_found -> false)
	then (if !debug then print_endline "COMPLETED"; completed := true)
	else ()
      else if (String.length l > 6 && (Str.first_chars l 4 = "LOCK" || Str.first_chars l 6 = "UNLOCK"))
      then begin
        output_string fd_trace_out l;
        output_char fd_trace_out '\n'
      end
      else if (String.length l > 2 && Str.first_chars l 2 = "IR")
      then begin
        output_string fd_trace_out "*IReads: ";
        output_string fd_trace_out
          (show_list (rewrite_indexes !index_map (ir_indexes_from_arm_run l)));
        output_char fd_trace_out '\n';
      end
      else
        begin
        index := !index +1;
        if (not (is_stack_accesses l) && (!trace_u_val || is_global_access addresses l))
	then begin
          trace_index := !trace_index + 1;
          index_map := (!index,!trace_index)::!index_map;
          output_string fd_trace_out l;
          output_char fd_trace_out '\n'
	end
        end
    done
  with End_of_file ->
    ignore (Unix.close_process (fdin,fdout));
    close_out fd_trace_out;
    ignore (Unix.close_process (fdin,fdout));
    if not !debug then Sys.remove objdump_name;
    if !completed then exit 0 else exit 1
